 /*
  * vdev_wait.c
  * Copyright (C) 2025  Aitor C.Z. <aitor_czr@gnuinos.org>
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */

#include "libvdev/sglib.h"
#include "libvdev/util.h"
#include "libvdev/misc.h"
#include "libvdev/sbuf.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdbool.h>
#include <sys/fcntl.h> 
#include <sys/ioctl.h>
#include <poll.h>
#include <signal.h>

#include <libinput.h> 
#include <libudev.h>
#include <libevdev/libevdev.h>

#include <mntent.h>
#include <sys/stat.h>

#include <poll.h>
#include <linux/netlink.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>

volatile sig_atomic_t stop_flag = 0;

typedef char *cstr;

// prototypes
SGLIB_DEFINE_VECTOR_PROTOTYPES (cstr);
SGLIB_DEFINE_VECTOR_FUNCTIONS (cstr);

const char *progname = "vdev_libinput_wait";
const char *mtab = "/etc/mtab";
const char *mountpoint = "/dev";
static char sysfs_mountpoint[PATH_MAX+1];

struct sysfs_scan_context {
    char *uevent_path;
    struct sglib_cstr_vector *frontier;
};

static void sigalrm_handler(int ignored) {
	stop_flag = 1;
}

// free a list of cstr vectors
// always succeeds
static int vdev_cstr_vector_free_all(struct sglib_cstr_vector *vec)
{
    // free all strings
    for (int i = 0; i < sglib_cstr_vector_size(vec); i++) {
        if (sglib_cstr_vector_at(vec, i) != NULL) {
            free(sglib_cstr_vector_at(vec, i));
            sglib_cstr_vector_set(vec, NULL, i);
        }
    }
    return 0;
}

static int open_restricted (const char *path, int flags, void *user_data)
{
	bool *grab = user_data;
	int fd = open(path, flags);

	if (fd < 0)
		fprintf(stderr, "Failed to open %s (%s)\n", path, strerror(errno));
	else if (grab && *grab && ioctl(fd, EVIOCGRAB, (void*)1) == -1)
		fprintf(stderr, "Grab requested, but failed for %s (%s)\n", path, strerror(errno));

	return fd < 0 ? -errno : fd;
}

static void close_restricted(int fd, void *user_data)
{
	close(fd);
}

static const struct libinput_interface interface = {
	.open_restricted = open_restricted,
	.close_restricted = close_restricted,
};

// 'buf' must be freed after usage
static void get_sysfs_path(char **buf)
{
    struct mntent *e;
    FILE *fstab = NULL;
   
    fstab = setmntent(mtab, "r");
    if (!fstab) {
        vdev_error("%s: setmntent(): error trying to open /etc/mtab: '%s'\n", progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
    
    *buf = (char*)malloc(sizeof(char) * 32);
    if (!*buf) {
        vdev_error ("%s: Memory allocation failure: '%s'\n", progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
   
    *buf[0] = '\0';
     
    while ((e = getmntent(fstab))) {
        if (!strcmp(e->mnt_type, "sysfs")) {
            sprintf(*buf, "%s", e->mnt_dir);
            break;
        }
    }
   
    endmntent(fstab);
}

int main(int argc, char **argv)
{
    int rc = 0;
    int fd = -1;	
		struct libinput *li = NULL;
		struct udev *udev = NULL;
    struct sglib_cstr_vector devices;
    bool started_removing_devices = false;
    struct stat statbuf;
    FILE *fp;
    const char *filename = "/dev/metadata/udev/input";

    sbuf_t s;
    char *buf = NULL;
    struct sglib_cstr_vector uevent_paths;
    struct sglib_cstr_vector names;
   
    if (access(mtab, F_OK) != 0) {
        snprintf (sysfs_mountpoint, PATH_MAX, "/sys"); 
    } else {
        get_sysfs_path (&buf);
        if (buf && buf[0] != '\0') {
            snprintf (sysfs_mountpoint, PATH_MAX, buf);       
            free (buf);   
        } else {
            vdev_error ("%s: Cannot get sysfs path\n", progname);
            exit (EXIT_FAILURE);
        }
    }
    
    if (access(filename, F_OK) == 0) {
        char line[1024];
        FILE *fp = fopen(filename, "r");
        if (fp == NULL)
            return -ENOSYS;

        sglib_cstr_vector_init(&names);
        while (fgets(line, 1024, fp)) {
            char *copy = NULL;
            line[strcspn(line, "\n")] = '\0';
            printf("\033[1m%s\e[0m\n", line);
            copy = vdev_strdup_or_null(line);
            if (!copy) {
                vdev_cstr_vector_free_all(&names);       
                sglib_cstr_vector_free(&names);
                fclose(fp);
                return -ENOMEM;
            }                
            rc = sglib_cstr_vector_push_back(&names, copy);
            if (rc < 0) {
                vdev_error ("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", progname, copy, strerror(errno));
                free(copy);
                vdev_cstr_vector_free_all(&names);       
                sglib_cstr_vector_free(&names);
                fclose(fp);
                return rc;
            }
        }
        fclose(fp);
    }
    
    if (sglib_cstr_vector_size(&names) == 0) {
        vdev_cstr_vector_free_all(&names);       
        sglib_cstr_vector_free(&names);
        return 0;
    }
	
    udev = udev_new();
    if (!udev) {
        fprintf(stderr, "Failed to initialize udev\n");
        return EXIT_FAILURE;
	}

	li = libinput_udev_create_context(&interface, NULL, udev);
	if (!li) {
        fprintf(stderr, "Failed to initialize context from udev\n");
        udev_unref (udev);
        return EXIT_FAILURE;
	}
	
	if (libinput_udev_assign_seat(li, "seat0")) {
		fprintf(stderr, "Failed to set seat\n");
		libinput_unref(li);
        udev_unref (udev);
        return EXIT_FAILURE;
	}	

    signal(SIGALRM, sigalrm_handler);
          
	//alarm(10);

    while (1) {
		int rv;
        bool shall_break = true;
		struct pollfd pfd = {libinput_get_fd(li), POLLIN, 0};
		rv = poll (&pfd, 1, 100);
		libinput_dispatch(li);
        struct libinput_event *ev;
		while ((ev = libinput_get_event(li))) {
            if (libinput_event_get_type(ev) == LIBINPUT_EVENT_DEVICE_ADDED) {
                struct libinput_device *dev = libinput_event_get_device(ev);
                for (int i = 0; i < sglib_cstr_vector_size(&names); i++) {
                    if (sglib_cstr_vector_at(&names, i) == NULL)
                        continue;
                    if (!strcmp(libinput_device_get_name(dev), sglib_cstr_vector_at(&names, i))) {
                        //printf("ADDED: %s\n", sglib_cstr_vector_at(&names, i));
                        sglib_cstr_vector_set(&names, NULL, i);
                        break;
                    }
                }
            }
            if (stop_flag == 1) {
                shall_break = true;               
            } else {
                for (int i = 0; i < sglib_cstr_vector_size(&names); i++) {
                    if (sglib_cstr_vector_at(&names, i) != NULL) {
                        shall_break = false;
                        break;
                    }
                }
            } 
            libinput_event_destroy(ev);     
            if (shall_break)
                break;
        }     
        if (shall_break)
            break;
	}

    vdev_cstr_vector_free_all(&names);       
    sglib_cstr_vector_free(&names);	
	
    libinput_unref(li);
	udev_unref(udev);

	return rc;
}
