  /*
  * policy.c
  * Copyright (C) Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
  * 
  * simple-netaid is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * simple-netaid is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */

#include <stdint.h>
#include <syslog.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>   // waitpid
#include <libgen.h>     // dirname() and basename()
#include <ctype.h>      // isdigit()	  
#include <errno.h>
#include <spawn.h>

#include <simple-netaid/sbuf.h>
#include <simple-netaid/ipaddr.h>
#include <simple-netaid/iproute.h>
#include <simple-netaid/interfaces.h>
#include <simple-netaid/helpers.h>

#include "def.h"
#include "config.h"
#include "policy.h"
#include "util.h"
#include "iwlist.h"

#include <json.h>
#include <libubox/blobmsg.h>
#include <libubox/blobmsg_json.h>
#include <libubox/json_script.h>

#define WIFI_DIR "/etc/network/wifi"
#define PREFIX "/tmp/simple-netaid_"

#define PTY_MAX_LEN 16

#define strdup_or_null( str )  (str) != NULL ? strdup(str) : NULL

static struct ubus_request_data req_data;
static struct blob_buf bb_t;

/* Enum for EGI policy order */
enum {
	INTERFACE_DOWN_ID,
	INTERFACE_DOWN_IFNAME,
	__INTERFACE_DOWN_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy interface_down_policy[] =
{
	[INTERFACE_DOWN_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[INTERFACE_DOWN_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	INTERFACE_UP_ID,
	INTERFACE_UP_IFNAME,
	__INTERFACE_UP_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy interface_up_policy[] =
{
	[INTERFACE_UP_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[INTERFACE_UP_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	IFDOWN_ID,
	IFDOWN_IFNAME,
	__IFDOWN_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ifdown_policy[] =
{
	[IFDOWN_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IFDOWN_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	IFUP_ID,
	IFUP_IFNAME,
	__IFUP_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ifup_policy[] =
{
	[IFUP_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IFUP_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	IPADDR_FLUSH_ID,
	IPADDR_FLUSH_IFNAME,
	__IPADDR_FLUSH_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ipaddr_flush_policy[] =
{
	[IPADDR_FLUSH_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IPADDR_FLUSH_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	DISCONNECT_ID,
	__DISCONNECT_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy disconnect_policy[] =
{
	[DISCONNECT_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	WIRED_CONNECTION_ID,
	WIRED_CONNECTION_IFNAME,
	__WIRED_CONNECTION_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wired_connection_policy[] =
{
	[WIRED_CONNECTION_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WIRED_CONNECTION_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};
 
/* Enum for EGI policy order */
enum {
	WPA_PASSPHRASE_ID,
	WPA_PASSPHRASE_ESSID,
	WPA_PASSPHRASE_PASSWD,
	WPA_PASSPHRASE_FILENAME,
	__WPA_PASSPHRASE_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wpa_passphrase_policy[] =
{
	[WPA_PASSPHRASE_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WPA_PASSPHRASE_ESSID] = { .name="essid", .type=BLOBMSG_TYPE_STRING },
	[WPA_PASSPHRASE_PASSWD] = { .name="passwd", .type=BLOBMSG_TYPE_STRING },
	[WPA_PASSPHRASE_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};
 
/* Enum for EGI policy order */
enum {
	WPA_SUPPLICANT_ID,
	WPA_SUPPLICANT_IFNAME,
	WPA_SUPPLICANT_FILENAME,
	__WPA_SUPPLICANT_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wpa_supplicant_policy[] =
{
	[WPA_SUPPLICANT_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WPA_SUPPLICANT_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[WPA_SUPPLICANT_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	WIRELESS_CONNECTION_ID,
	WIRELESS_CONNECTION_IFNAME,
	WIRELESS_CONNECTION_ESSID,
	WIRELESS_CONNECTION_PASSWD,
	WIRELESS_CONNECTION_FILENAME,
	__WIRELESS_CONNECTION_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wireless_connection_policy[] =
{
	[WIRELESS_CONNECTION_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WIRELESS_CONNECTION_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_ESSID] = { .name="essid", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_PASSWD] = { .name="passwd", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	UNINSTALL_SAVED_WIFI_ID,
	UNINSTALL_SAVED_WIFI_FILENAME,
	__UNINSTALL_SAVED_WIFI_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy uninstall_saved_wifi_policy[] =
{
	[UNINSTALL_SAVED_WIFI_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[UNINSTALL_SAVED_WIFI_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	SCAN_ACTIVE_WIFIS_ID,
	SCAN_ACTIVE_WIFIS_IFNAME,
	SCAN_ACTIVE_WIFIS_FORMAT,
	__SCAN_ACTIVE_WIFIS_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy scan_active_wifis_policy[] =
{
	[SCAN_ACTIVE_WIFIS_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[SCAN_ACTIVE_WIFIS_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[SCAN_ACTIVE_WIFIS_FORMAT]  = { .name="format", .type=BLOBMSG_TYPE_INT32 },
};

 
/* Ubus Methods */
const struct ubus_method netaid_methods[] =
{
	UBUS_METHOD("interface_down", interface_down_handler, interface_down_policy),
	UBUS_METHOD("interface_up", interface_up_handler, interface_up_policy),
	UBUS_METHOD("ifdown", ifdown_handler, ifdown_policy),
	UBUS_METHOD("ifup", ifup_handler, ifup_policy),
	UBUS_METHOD("ipaddr_flush", ipaddr_flush_handler, ipaddr_flush_policy),
	UBUS_METHOD("disconnect", disconnect_handler, disconnect_policy),
	UBUS_METHOD("wired_connection", wired_connection_handler, wired_connection_policy),
	UBUS_METHOD("wpa_passphrase", wpa_passphrase_handler, wpa_passphrase_policy),
	UBUS_METHOD("wpa_supplicant", wpa_supplicant_handler, wpa_supplicant_policy),
	UBUS_METHOD("wireless_connection", wireless_connection_handler, wireless_connection_policy),
	UBUS_METHOD("uninstall_saved_wifi", uninstall_saved_wifi_handler, uninstall_saved_wifi_policy),
	UBUS_METHOD("scan_active_wifis", scan_active_wifis_handler, scan_active_wifis_policy),
};
 
/* Ubus object type */
struct ubus_object_type netaid_obj_type =
	UBUS_OBJECT_TYPE("ering.netaid", netaid_methods);
 
/* Ubus object */
struct ubus_object netaid_obj=
{
	.name = "ering.netaid", 	/* with APP name */
	.type = &netaid_obj_type,
	.methods = netaid_methods,
	.n_methods = ARRAY_SIZE(netaid_methods),
	//.path= /* useless */
};

/**
 * \brief Callback function for ubus interface_down method handling
 */
int interface_down_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg_t)
{
    char reply_msg[128];
    struct blob_attr *tb[__INTERFACE_DOWN_MAX];
    
    blobmsg_parse(interface_down_policy, ARRAY_SIZE(interface_down_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[INTERFACE_DOWN_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[INTERFACE_DOWN_IFNAME]);
   
    /* Network logic */
    /* interface_down() must call ioctl/netlink */
    interface_down(ifname);

    /* Response to Ubus using static buffer */
    snprintf(reply_msg, sizeof(reply_msg), "ip link set %s down", ifname);
 
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
    ubus_send_reply(ctx, req, bb_t.head);
 
    /* Request completion */
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    return 0;
}

/**
 * \brief Callback function for ubus interface_up method handling
 */
int interface_up_handler(struct ubus_context *ctx, struct ubus_object *obj,
                         struct ubus_request_data *req, const char *method,
                         struct blob_attr *msg_t)
{
   char reply_msg[128];
   struct blob_attr *tb[__INTERFACE_UP_MAX];
    
   blobmsg_parse(interface_up_policy, ARRAY_SIZE(interface_up_policy), tb, blob_data(msg_t), blob_len(msg_t));

   if (!tb[INTERFACE_UP_IFNAME])
      return UBUS_STATUS_INVALID_ARGUMENT;

   const char *ifname = blobmsg_get_string(tb[INTERFACE_UP_IFNAME]);
   
   interface_up(ifname);

   snprintf(reply_msg, sizeof(reply_msg), "ip link set %s up", ifname);
 
   blob_buf_init(&bb_t, 0);
   blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
   ubus_send_reply(ctx, req, bb_t.head);
 
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   return 0;
}

/**
 * \brief Callback function for ubus ifdown method handling
 */
int ifdown_handler(struct ubus_context *ctx, struct ubus_object *obj,
                   struct ubus_request_data *req, const char *method,
                   struct blob_attr *msg_t)
{
    int rc = 0;
    char reply_msg[64];
    bool redirect_ok = false;
    struct blob_attr *tb[__IFDOWN_MAX];
 
    blobmsg_parse(ifdown_policy, ARRAY_SIZE(ifdown_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[IFDOWN_IFNAME])
       return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IFDOWN_IFNAME]);   

    if (!ifquery(ifname)) 
       return 0;
   
    char *const ifdown_args[] = { "ifdown", "--force", (char *)ifname, NULL };
    rc = safe_execute(ifdown_args);
   
    if (!redirect_ok) {
        snprintf(reply_msg, sizeof(reply_msg), "ifdown %s", ifname);
 
        blob_buf_init(&bb_t, 0);
        blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
        ubus_send_reply(ctx, req, bb_t.head);
    }
   
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    return rc;
}

/**
 * \brief Callback function for ubus ifup method handling (Optimizado para ifupdown2)
 */
int ifup_handler(struct ubus_context *ctx, struct ubus_object *obj,
                 struct ubus_request_data *req, const char *method,
                 struct blob_attr *msg_t)
{
    int rc = -1;
    struct blob_attr *tb[__IFUP_MAX];

    blobmsg_parse(ifup_policy, ARRAY_SIZE(ifup_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[IFUP_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IFUP_IFNAME]);
   
    char *const ifup_args[] = { "ifup", (char *)ifname, NULL };
    int wstatus = safe_execute(ifup_args); 

    blob_buf_init(&bb_t, 0);
    if (wstatus == 0) {
        rc = 0;
        blobmsg_add_string(&bb_t, "result", "success");
        blobmsg_add_string(&bb_t, "message", "Interface is up");
    } else {
        blobmsg_add_string(&bb_t, "result", "error");
        blobmsg_add_u32(&bb_t, "exit_code", wstatus);
    }

    ubus_send_reply(ctx, req, bb_t.head);
   
    return rc;
}

/**
 * \brief Callback function for ubus ipaddr_flush method handling
 */
int ipaddr_flush_handler(struct ubus_context *ctx, struct ubus_object *obj,
                         struct ubus_request_data *req, const char *method,
                         struct blob_attr *msg_t)
{
    char wired_if[IFNAMSIZ] = {0};
    char wireless_if[IFNAMSIZ] = {0};

    struct blob_attr *tb[__IPADDR_FLUSH_MAX];
 
    blobmsg_parse(ipaddr_flush_policy, ARRAY_SIZE(ipaddr_flush_policy), tb, blob_data(msg_t), blob_len(msg_t));
   
    if (!tb[IPADDR_FLUSH_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IPADDR_FLUSH_IFNAME]);

    get_interfaces_simple(wired_if, sizeof(wired_if), wireless_if, sizeof(wireless_if));
   
    if (wireless_if[0] && strcmp(ifname, wireless_if) == 0) {
        if (wired_if[0] && get_interface_status(wired_if)) {
            interface_down(wired_if);
            usleep(2500);
            interface_up(wired_if);
        }
    } else if (wired_if[0] && strcmp(ifname, wired_if) == 0) {
        if (wireless_if[0] && get_interface_status(wireless_if)) {
            interface_down(wireless_if);
            usleep(2500);
            interface_up(wireless_if);
        }
    }
   
    sleep(1);
    kill_all_processes(ifname, NULL);
    ipaddr_flush(ifname);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "status", "flushed");
    ubus_send_reply(ctx, req, bb_t.head);

    return 0;
}

/**
 * \brief Callback function for ubus disconnect method handling
 */
int disconnect_handler(struct ubus_context *ctx, struct ubus_object *obj,
                       struct ubus_request_data *req, const char *method,
                       struct blob_attr *msg_t)
{
   int rc = 0;
   char reply_msg[128] = "Disconnected";
   int wstatus;
   bool redirect_ok = false;
   
   struct blob_attr *tb[__DISCONNECT_MAX];
   blobmsg_parse(disconnect_policy, ARRAY_SIZE(disconnect_policy), tb, blob_data(msg_t), blob_len(msg_t));

   /* Detecting the init system (Runit) */
   /*
   char buf[32] = {0};
   ssize_t sz = readlink("/proc/1/exe", buf, sizeof(buf) - 1);
   if (sz != -1) {
       buf[sz] = '\0';
       if (strstr(buf, "runit")) {
           char *const stop_dhcp[] = { "sv", "stop", "dhcpcd", NULL };
           safe_execute(stop_dhcp);
       }
   }
   */
   
   const char *ifname = iproute();
   if (ifname && *ifname != '0' && *ifname != '\0') {
       snprintf(reply_msg, sizeof(reply_msg), "Disconnected from %s", ifname);

        // Cleaning up ifupdown2 (Normal + Force)
        char *const down_args[] = { "ifdown", "-v", (char *)ifname, NULL };
        if (safe_execute(down_args) != 0) {
            char *const force_args[] = { "ifdown", "-v", "--force", (char *)ifname, NULL };
            safe_execute(force_args);
        }
  
       interface_down(ifname);       
       ipaddr_flush(ifname);
       kill_all_processes(ifname, NULL);
       interface_up(ifname);
   } else {
       strlcpy(reply_msg, "No active connection to disconnect", sizeof(reply_msg));
   }
   
   if (!redirect_ok) {
       blob_buf_init(&bb_t, 0);
       blobmsg_add_string(&bb_t, "reply", reply_msg);
       ubus_send_reply(ctx, req, bb_t.head);
   }
    
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   return rc;
}

/**
 * \brief Callback function for ubus wired_connection method handling
 */
int wired_connection_handler(struct ubus_context *ctx, struct ubus_object *obj,
                             struct ubus_request_data *req, const char *method,
                             struct blob_attr *msg_t)
{ 
    int rc = -1;
    char reply_msg[128] = {0};
    int wstatus;
    bool redirect_ok = false;
   
    const char *current_ip = iproute();
    if (current_ip && *current_ip != '0' && *current_ip != '\0') {
        snprintf(reply_msg, sizeof(reply_msg), "Already connected to %s", current_ip);
        goto Finish_wired_connection;
    }
   
    struct blob_attr *tb[__WIRED_CONNECTION_MAX];
    blobmsg_parse(wired_connection_policy, ARRAY_SIZE(wired_connection_policy), tb, blob_data(msg_t), blob_len(msg_t)); 
  
    if (!tb[WIRED_CONNECTION_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[WIRED_CONNECTION_IFNAME]);
   
    if (get_interface_status(ifname)) {
        interface_down(ifname);
        usleep(2500);
        interface_up(ifname);
        sleep(1);
    }

    kill_all_processes(ifname, NULL);
    ipaddr_flush(ifname);
    interface_down(ifname);
   
    if (ifquery(ifname)) { 
        char *const down_args[] = { "ifdown", (char *)ifname, NULL };
        if (safe_execute(down_args) != 0) {
            char *const force_args[] = { "ifdown", "--force", (char *)ifname, NULL };
            if (safe_execute(force_args) != 0)
                goto Finish_wired_connection;
        }
    }
   
    interface_up(ifname);
    sleep(1);

   if (ifquery(ifname) == 0) {
        char *const clean_args[] = { "/sbin/ifdown", "--force", (char *)ifname, NULL };
        safe_execute(clean_args);

        char *const ifup_cmd[] = { "/sbin/ifup", (char *)ifname, NULL };
        wstatus = safe_execute(ifup_cmd);
    
        if (wstatus == -1) {
            snprintf(reply_msg, sizeof(reply_msg), "System error: %s", strerror(errno));
        } else {
            rc = 0;
            snprintf(reply_msg, sizeof(reply_msg), "Interface %s is UP", ifname);
        }
    }

Finish_wired_connection:

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "reply", *reply_msg ? reply_msg : "Connection failed");
    ubus_send_reply(ctx, req, bb_t.head);
   
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    return rc;
}

/**
 * \brief Callback function for ubus wpa_passphrase method handling
 */
int wpa_passphrase_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg)
{
    struct blob_attr *tb[__WPA_PASSPHRASE_MAX];
    blobmsg_parse(wpa_passphrase_policy, __WPA_PASSPHRASE_MAX, tb, blob_data(msg), blob_len(msg));

    if (!tb[WPA_PASSPHRASE_ESSID] || !tb[WPA_PASSPHRASE_PASSWD] || !tb[WPA_PASSPHRASE_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *essid = blobmsg_get_string(tb[WPA_PASSPHRASE_ESSID]);
    const char *passwd = blobmsg_get_string(tb[WPA_PASSPHRASE_PASSWD]);
    const char *path = blobmsg_get_string(tb[WPA_PASSPHRASE_FILENAME]);

    char *const args[] = { "wpa_passphrase", (char *)essid, (char *)passwd, NULL };
    int res = safe_execute_to_file(path, false, args);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "status", (res == 0) ? "created" : "failed");
    ubus_send_reply(ctx, req, bb_t.head);

    return 0;
}

/**
 * \brief Callback function for ubus wpa_supplicant method handling
 */
int wpa_supplicant_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg_t)
{
    int rc = -1;     
    int wstatus;
    struct blob_attr *tb[__WPA_SUPPLICANT_MAX];
 
    blobmsg_parse(wpa_supplicant_policy, ARRAY_SIZE(wpa_supplicant_policy), tb, blob_data(msg_t), blob_len(msg_t));   
   
    if (!tb[WPA_SUPPLICANT_IFNAME] || !tb[WPA_SUPPLICANT_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[WPA_SUPPLICANT_IFNAME]);
    const char *filename = blobmsg_get_string(tb[WPA_SUPPLICANT_FILENAME]);

    char *const wpa_args[] = {
        "wpa_supplicant",
        "-B",
        "-i", (char *)ifname,
        "-c", (char *)filename,
        NULL
    };

    wstatus = safe_execute(wpa_args);
    
    if (wstatus == 0) {
        rc = 0;
    } else {
        syslog(LOG_ERR,
            "Failure trying to start wpa_supplicant in %s with config %s", ifname, filename);
    }

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "result", (rc == 0) ? "success" : "failed");
    blobmsg_add_string(&bb_t, "command", "wpa_supplicant");
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    return rc;
}

/**
 * \brief Callback function for ubus wireless_connection method handling
 */
int wireless_connection_handler(struct ubus_context *ctx, struct ubus_object *obj,
                               struct ubus_request_data *req, const char *method,
                               struct blob_attr *msg_t)
{ 
    int rc = -1;     
    char path[1024];
    char reply_msg[256] = {0};
    int wstatus = 0;
    bool pass_filter_ok = false;
   
    const char *current_ip = iproute();
    if (current_ip && *current_ip != '0' && *current_ip != '\0') {
        snprintf(reply_msg, sizeof(reply_msg), "Already connected to %s", current_ip);
        goto Finish_wireless_connection;
    }
   
    struct blob_attr *tb[__WIRELESS_CONNECTION_MAX];
    blobmsg_parse(wireless_connection_policy, ARRAY_SIZE(wireless_connection_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (!tb[INTERFACE_UP_IFNAME] || !tb[WIRELESS_CONNECTION_ESSID])
        return UBUS_STATUS_INVALID_ARGUMENT;
  
    const char *ifname = blobmsg_get_string(tb[INTERFACE_UP_IFNAME]);
    const char *essid = blobmsg_get_string(tb[WIRELESS_CONNECTION_ESSID]);
    const char *passwd = tb[WIRELESS_CONNECTION_PASSWD] ? blobmsg_get_string(tb[WIRELESS_CONNECTION_PASSWD]) : "";
    const char *filename = tb[WIRELESS_CONNECTION_FILENAME] ? blobmsg_get_string(tb[WIRELESS_CONNECTION_FILENAME]) : "";
	  
    if (filename && *filename != '\0') {
        snprintf(path, sizeof(path), "%s", filename);
    } else {
        unsigned int r = (unsigned int)time(NULL) % 10000;
        snprintf(path, sizeof(path), "/tmp/simple-netaid_%u", r);
    }
    
    char *tmp_path_ptr = strdup_or_null(path);
    if (tmp_path_ptr) {
        char *dname = dirname(tmp_path_ptr);
        if (dname && (strcmp(dname, "/etc/network/wifi") == 0 ||
                strncmp(path, "/tmp/simple-netaid_", 19) == 0)) {
            pass_filter_ok = true;
        }
        free(tmp_path_ptr);
    }
   
    kill_all_processes(ifname, NULL);
    ipaddr_flush(ifname);   
    interface_down(ifname);
    
    if (ifquery(ifname)) {
        char *const ifdown_cmd[] = { "ifdown", "-v", (char *)ifname, NULL };
        safe_execute(ifdown_cmd);    
    }
   
    interface_up(ifname);
   
    if (pass_filter_ok && *passwd != '\0' && access(path, F_OK) != 0) {
        char *const pass_args[] = { "wpa_passphrase", (char *)essid, (char *)passwd, NULL };
        wstatus = safe_execute_to_file(path, false, pass_args);
        if (wstatus != 0) goto Finish_wireless_connection;
    } 
    else if (*passwd == '\0' && (!filename || *filename == '\0')) {
        int fd = open(path, O_CREAT | O_WRONLY | O_TRUNC, 0600);
        if (fd >= 0) {
            dprintf(fd, "network={\n\tssid=\"%s\"\n\tkey_mgmt=NONE\n}\n", essid);
            close(fd);
        } else {
            goto Finish_wireless_connection;
        }
    }

    char *const wpa_args[] = { "wpa_supplicant", "-B", "-i", (char *)ifname, "-c", path, NULL };
    if (safe_execute(wpa_args) != 0) goto Finish_wireless_connection;      
   
    sleep(1);

    char *const ifup_cmd[] = { "ifup", "-v", (char *)ifname, NULL };
    wstatus = safe_execute(ifup_cmd);

    if (wstatus == 0) {
        rc = 0;
        snprintf(reply_msg, sizeof(reply_msg), "Interface %s is UP and configured", ifname);
    } else {
        snprintf(reply_msg, sizeof(reply_msg), "ifupdown2 failed (exit code %d)", wstatus);
    }

Finish_wireless_connection:
    if (rc != 0 && (!filename || *filename == '\0'))
        unlink(path);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "reply", *reply_msg ? reply_msg : "Operation failed");
    ubus_send_reply(ctx, req, bb_t.head);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    return rc;
}

/**
 * \brief Callback function for ubus uninstall_saved_wifi method handling
 */
int uninstall_saved_wifi_handler(struct ubus_context *ctx, struct ubus_object *obj,
                      struct ubus_request_data *req, const char *method,
                      struct blob_attr *msg_t)
{
    struct blob_attr *tb[__UNINSTALL_SAVED_WIFI_MAX];
    blobmsg_parse(uninstall_saved_wifi_policy, __UNINSTALL_SAVED_WIFI_MAX, tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[UNINSTALL_SAVED_WIFI_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *filename = blobmsg_get_string(tb[UNINSTALL_SAVED_WIFI_FILENAME]);

    if (strstr(filename, "..") ||
        !is_wpa_conf_file(filename) ||
        (strncmp(filename, WIFI_DIR, strlen(WIFI_DIR)) != 0 && 
                                   strncmp(filename, PREFIX, strlen(PREFIX)) != 0))
        return UBUS_STATUS_PERMISSION_DENIED;

    int rc = unlink(filename);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "result", (rc == 0) ? "deleted" : "failed");
    ubus_send_reply(ctx, req, bb_t.head);
    blob_buf_free(&bb_t);
    return (rc == 0) ? UBUS_STATUS_OK : UBUS_STATUS_UNKNOWN_ERROR;
}

/**
 * \brief Callback function for ubus scan_active_wifis method handling
 */
int scan_active_wifis_handler(struct ubus_context *ctx, struct ubus_object *obj,
                              struct ubus_request_data *req, const char *method,
                              struct blob_attr *msg_t)
{
    struct blob_attr *tb[__SCAN_ACTIVE_WIFIS_MAX];
    struct blob_buf bb_t = {0};
    int count;

    blobmsg_parse(scan_active_wifis_policy, ARRAY_SIZE(scan_active_wifis_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[SCAN_ACTIVE_WIFIS_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    blob_buf_init(&bb_t, 0);

    count = iwlist(blobmsg_get_string(tb[SCAN_ACTIVE_WIFIS_IFNAME]), &bb_t);
    if (count < 0) {
        blob_buf_free(&bb_t);
        return UBUS_STATUS_UNKNOWN_ERROR;
    }

    ubus_send_reply(ctx, req, bb_t.head);
    blob_buf_free(&bb_t);

    return UBUS_STATUS_OK;
}
