/*
 * netaid-window.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include <gtk/gtk.h>
#include <vte/vte.h>

// Required for GDK_PROPERTY_CHANGE_MASK if not included by default
#include <gdk/gdkx.h>

#include "netaid-window.h"
#include "netaid-config.h"
#include "netaid-status-icon.h"
#include "netaid-menu-bar.h"
#include "netaid-toolbar.h"
#include "netaid-notebook.h"
#include "netaid-active-wifis.h"

#include <stdbool.h>
#include <math.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>		/* write, pipe */
#include <fcntl.h>		/* fcntl, O_NONBLOCK */

#include <simple-netaid/sbuf.h>
#include <simple-netaid/iproute.h>

#define WIN_WIDTH 750
#define WIN_HEIGHT 600

Atom atom_desktop_x11;

/* Forward declarations */
GdkPixbuf *get_app_icon();
GdkPixbuf *get_icon_connected();
GdkPixbuf *get_icon_disconnected();

int userinfo_init(void);
const char *userhome(void);

static void move_x11_window_to_desktop(GtkWindow *window, long desktop_index)
{
    GdkWindow *gdk_win = gtk_widget_get_window(GTK_WIDGET(window));
    if (!gdk_win) return;

    Window xid = GDK_WINDOW_XID(gdk_win);
    Display *display = GDK_DISPLAY_XDISPLAY(gdk_display_get_default());
    Window root = DefaultRootWindow(display);

    Atom atom_move = XInternAtom(display, "_NET_WM_DESKTOP", False);

    XEvent xev;
    xev.type = ClientMessage;
    xev.xclient.window = xid;
    xev.xclient.message_type = atom_move;
    xev.xclient.format = 32;
    xev.xclient.data.l[0] = desktop_index;
    xev.xclient.data.l[1] = 1;
    xev.xclient.data.l[2] = 0;
    xev.xclient.data.l[3] = 0;
    xev.xclient.data.l[4] = 0;

    XSendEvent(display, root, False, 
               SubstructureNotifyMask | SubstructureRedirectMask, &xev);
    
    XFlush(display);
}


static GdkFilterReturn my_x11_filter(GdkXEvent *xevent, GdkEvent *event, gpointer user_data)
{
    XEvent *xev = (XEvent *)xevent;
    NetaidWindow *self = (NetaidWindow *)user_data;
    
    Display *display = self->display;

    if (xev->type == PropertyNotify && xev->xproperty.atom == atom_desktop_x11) {
        Atom curr_type;
        int curr_format;
        unsigned long nitems, bytes_after;
        unsigned char *prop_value = NULL;
        Window root = DefaultRootWindow(display);

        if (XGetWindowProperty(display, root, atom_desktop_x11, 0, 1, False, XA_CARDINAL,
                               &curr_type, &curr_format, &nitems, &bytes_after,
                               &prop_value) == Success && prop_value) {
            
            long current_desktop = *(long *)prop_value;
            g_print("GDK/X11: Escritorio cambiado a: %ld\n", current_desktop);
            move_x11_window_to_desktop(GTK_WINDOW(self), current_desktop);   
            XFree(prop_value);
        }
    }
    return GDK_FILTER_CONTINUE;
}

/*
 * Unix signals that are cought are written to a pipe. The pipe connects
 * the unix signal handler with GTK's event loop. The array signal_pipe will
 * hold the file descriptors for the two ends of the pipe(index 0 for
 * reading, 1 for writing).
 * As reaction to a unix signal we change the text of a label, hence the
 * label must be global.
 */
int signal_pipe[2];
static int rc_int_val;

static void _paned_change_handle_size_green(GtkWidget *widget)
{
    GtkStyleContext *context = gtk_widget_get_style_context(widget);

    GtkCssProvider *css = gtk_css_provider_new();
    gtk_css_provider_load_from_data(
            css,
            "paned separator{"
            "background-size:3px;"
            "background-color:#5a7100;"
            "min-height: 6px;"
            "}",
            -1
            ,NULL
	);

    gtk_style_context_add_provider(context, GTK_STYLE_PROVIDER(css),
            GTK_STYLE_PROVIDER_PRIORITY_USER);
}

static char *read_link(const char *filename)
{
    char *buf;
    char *newbuf;
    size_t cap;
    ssize_t len;

    buf = NULL;
    for (cap = 64; cap <= 16384; cap *= 2) {
        newbuf =(char*)realloc(buf, cap);
        if (newbuf == NULL) {
            break;
        }
        buf = newbuf;
        len = readlink(filename, buf, cap);
        if (len < 0) {
            break;
        }
        if ((size_t)len < cap) {
            buf[len] = 0;
            return buf;
        }
    }

    free(buf);
    return NULL;
}

/* A helper function that takes variable arguments and uses g_string_vprintf */
void string_format(GString *g_str, const gchar *format, ...)
{
    va_list args;
    va_start(args, format);
    // Use g_string_vprintf to format the string from the va_list
    g_string_vprintf(g_str, format, args);
    va_end(args);
}

static void sig_handler(int signum, siginfo_t *info, void *extra)
{
    switch (signum) {
    case SIGUSR1:
    case SIGUSR2:
         rc_int_val = info->si_value.sival_int;
         if (write(signal_pipe[1], &signum, sizeof(int)) != sizeof(int))
            fprintf(stderr, "linux signal %d lost\n", signum);
         break;
    case SIGTERM:
    case SIGINT:
    case SIGQUIT:
         break;
   }
}

/*
 * The event loop callback that handles the unix signals. Must be a GIOFunc.
 * The source is the reading end of our pipe, cond is one of
 * G_IO_IN or G_IO_PRI(I don't know what could lead to G_IO_PRI)
 * the pointer d is always NULL
 */
gboolean deliver_signal(GIOChannel *source, GIOCondition cond, gpointer user_data)
{
    NetaidWindow *window =(NetaidWindow *)user_data;
    GError *error = NULL;
    GIOStatus status;		/* save the reading status */
    gsize bytes_read;		/* save the number of chars read */

    /*
     * There is no g_io_channel_read or g_io_channel_read_int, so we read
     * char's and use a union to recover the unix signal number.
     */
    union {
        gchar chars[sizeof(int)];
        int signal;
    } buf;

    /*
     * Read from the pipe as long as data is available. The reading end is
     * also in non-blocking mode, so if we have consumed all unix signals,
     * the read returns G_IO_STATUS_AGAIN.
     */
    while ((status = g_io_channel_read_chars(source, buf.chars,
			sizeof(int), &bytes_read, &error)) == G_IO_STATUS_NORMAL) {
        
        g_assert(error == NULL);	/* no error if reading returns normal */

        /*
         * There might be some problem resulting in too few char's read.
         * Check it.
         */
        if (bytes_read != sizeof(int)) {
            fprintf(stderr, "lost data in signal pipe(expected %ld, received %ld)\n",
                        sizeof(int), bytes_read);
            /* discard the garbage and keep fingers crossed */
            continue;
        }

        /* Ok, we read a linux signal number */
        if (buf.signal == SIGUSR1 /* && rc_int_val == 1 */) {
            if (*iproute() == 0)
                window->connected = FALSE;
            else
                window->connected = TRUE;
        } else if (buf.signal == SIGUSR2) {
            if (*iproute() == 0)
                window->connected = FALSE;
            else
                window->connected = TRUE;
            if (rc_int_val == 2)
                g_print("The ethernet cable has been plugged in\n");
            else if (rc_int_val == 3)
                g_print("Connect to wireless device\n");
            else if (rc_int_val < 0 || rc_int_val > 3)
                g_print("Change the interval\n");
        }
    }

    /*
     * Reading from the pipe has not returned with normal status. Check for
     * potential errors and return from the callback.
     */
    if (error != NULL) {
        fprintf(stderr, "reading signal pipe failed: %s\n", error->message);
        exit(1);
    }

    if (status == G_IO_STATUS_EOF) {
        fprintf(stderr, "signal pipe has been closed\n");
        exit(1);
    }

    g_assert(status == G_IO_STATUS_AGAIN);

    /* keep the event source */
    return(TRUE);
}

typedef struct {
    gboolean systray_ok;
} NetaidWindowPrivate;

enum {
    PROP_0,
    PROP_CONNECTED,
    PROP_ICONIFIED,
    PROP_IFNAME,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = {NULL, };

G_DEFINE_TYPE_WITH_PRIVATE(NetaidWindow, netaid_window, GTK_TYPE_APPLICATION_WINDOW)

NetaidWindow *get_netaid_window_from_xid(Window xid)
{
    GdkDisplay *display = gdk_display_get_default();
    
    GdkWindow *gdk_win = gdk_x11_window_lookup_for_display(display, xid);
    
    if (gdk_win) {
        gpointer user_data = NULL;
        gdk_window_get_user_data(gdk_win, &user_data);
        
        if (user_data && NETAID_IS_WINDOW(user_data)) {
            return NETAID_WINDOW(user_data);
        }
    }
    return NULL;
}

// Signal handler for the "notify::connected"
static void on_ifname_property_notify_event(GObject *gobject,
		GParamSpec *pspec, gpointer user_data)
{
    g_return_if_fail(NETAID_IS_WINDOW(gobject));
    
    NetaidWindow *self = NETAID_WINDOW(gobject);

	char *markup;    
    const char *name;
    gboolean is_connected;
   
    name = pspec->name;
    g_object_get(G_OBJECT(self), name, &is_connected, NULL);
    
    if (self->connected) {
		GString *buf = g_string_new(NULL);
		g_string_append(buf, " Connected to ");
		g_string_append(buf, self->ifname->str);
		const char *format = "<span background=\"green\" foreground=\"white\"> %s</span>";
		markup = g_markup_printf_escaped(format, buf->str);
		g_string_free(buf, TRUE);
	}
    else {
		const char *format = "<span background=\"red\" foreground=\"white\"> %s</span>";
		markup = g_markup_printf_escaped(format, " Disconnected");
	}  
	
	gtk_label_set_markup(GTK_LABEL(self->label), markup);
	g_free(markup);
}

static void netaid_window_set_property(GObject *object,
		guint property_id, const GValue *value, GParamSpec *pspec)
{
    NetaidWindow *self = NETAID_WINDOW(object);
    
    switch (property_id) {
    case PROP_CONNECTED:
        self->connected = g_value_get_boolean(value);
        break;
    case PROP_ICONIFIED:
        self->iconified = g_value_get_boolean(value);
        break;
    case PROP_IFNAME:
        /* We don't need to trigger the event with g_object_notify,
         * because we are not passing G_PARAM_EXPLICIT_NOTIFY to 
         * g_param_spec_string() in netaid_window_class_init */
        g_string_assign(self->ifname, g_value_get_string(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
        break;
    }
}

static void netaid_window_get_property(GObject *object,
		guint property_id, GValue *value, GParamSpec *pspec)
{
    NetaidWindow *self = NETAID_WINDOW(object);
    
    switch (property_id) {
    case PROP_CONNECTED:
        g_value_set_boolean(value, self->connected);
        break;
    case PROP_ICONIFIED:
        g_value_set_boolean(value, self->iconified);
        break;
    case PROP_IFNAME:
		g_value_set_string(value, self->ifname->str);
		break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
        break;
    }
}

static void netaid_window_finalize(GObject *object)
{
    NetaidWindow *self = NETAID_WINDOW(object);
    
    if (self->timer_handler_id > 0) {
        g_source_remove(self->timer_handler_id);
        self->timer_handler_id = 0;
    }
    
    if (self->netaid_config) {
        g_object_unref (self->netaid_config);
        self->netaid_config = NULL;
    }

    /* netaid_window is not ancestor of status_icon */
    if (self->status_icon) {
        g_object_unref (self->status_icon);
        self->status_icon = NULL;
    }
    
    if (self->ifname) {
        g_string_free(self->ifname, TRUE);
        self->ifname = NULL;
    }

    /* chain to the parent */
    G_OBJECT_CLASS(netaid_window_parent_class)->finalize(object);
}

static void netaid_window_class_init(NetaidWindowClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    gobject_class->finalize = netaid_window_finalize;
    gobject_class->set_property = netaid_window_set_property;
    gobject_class->get_property = netaid_window_get_property;

    properties[PROP_CONNECTED] = g_param_spec_boolean("connected", NULL, NULL,
            TRUE,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS
            /* | G_PARAM_EXPLICIT_NOTIFY */);

    properties[PROP_ICONIFIED] = g_param_spec_boolean("iconified", NULL, NULL,
            TRUE,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS
            /* | G_PARAM_EXPLICIT_NOTIFY */);

    properties[PROP_IFNAME] = g_param_spec_string("ifname", NULL, NULL, NULL,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

    g_object_class_install_properties(gobject_class, N_PROPERTIES, properties);
}

static gboolean time_handler(gpointer user_data)
{
    g_return_val_if_fail(NETAID_IS_WINDOW(user_data), FALSE);
    
    NetaidWindow *self = NETAID_WINDOW(user_data);

    NetaidWindowPrivate *priv = netaid_window_get_instance_private(self);

    const char *ifname = iproute();
    gboolean is_connected = (ifname[0] != '\0');
    
    const char *text = gtk_label_get_text(GTK_LABEL(self->label));
    
    if (text[0] == '\0') {
        self->connected = is_connected;
        g_object_notify(G_OBJECT(self), "connected");
        g_object_notify(G_OBJECT(self), "ifname");
        if (priv->systray_ok)
            netaid_window_status_icon_update(self);
        goto Finish;
    }

    if (self->connected != is_connected) {
        self->connected = is_connected;
        g_object_notify(G_OBJECT(self), "connected");    
        if (!is_connected) {
            g_string_truncate(self->ifname, 0);
            g_object_notify(G_OBJECT(self), "ifname");
        }
        if (priv->systray_ok)
            netaid_window_status_icon_update(self);
    }
    
    if (is_connected) {
        if (self->ifname && !strcmp(ifname, self->ifname->str))
            goto Finish;
        g_string_assign(self->ifname, ifname);
        g_object_notify(G_OBJECT(self), "ifname");
        if (priv->systray_ok)
            netaid_window_status_icon_update(self);
    }
    
Finish:    
    return G_SOURCE_CONTINUE;
}

static void netaid_window_init(NetaidWindow *self)
{
    GString *g_str;
    GtkWidget *term1, *term2;
    GtkWidget *pane1, *pane2;
    GtkWidget *box1, *box2;
	char *markup;
	const char *format;
    NetaidWindowPrivate *priv = netaid_window_get_instance_private(self);
    
    self->systray_ok = TRUE;
    self->ifname = g_string_new(NULL);

    self->netaid_config = netaid_config_new();

    /*
     * In order to register the reading end of the pipe with the event loop
     * we must convert it into a GIOChannel
     */
    GIOChannel *g_signal_in;
    GError *error = NULL;   /* handle errors */
    long fd_flags; 	        /* used to change the pipe into non-blocking mode */
    struct sigaction sa;

    GdkPixbuf *pPix = get_app_icon();
    gtk_window_set_icon(GTK_WINDOW(self), pPix);

    gtk_container_set_border_width(GTK_CONTAINER(self), 1);

    box1 = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    box2 = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);

    pane1 = gtk_paned_new(GTK_ORIENTATION_VERTICAL);
    pane2 = gtk_paned_new(GTK_ORIENTATION_VERTICAL);

	self->label = gtk_label_new(NULL);
    g_object_set_data(G_OBJECT(self), "label", self->label);

    self->menu_bar = netaid_menu_bar_new();
    self->toolbar = netaid_toolbar_new();
    self->notebook = netaid_notebook_new();

    gtk_container_add(GTK_CONTAINER(self), box1);
    gtk_box_pack_start(GTK_BOX(box1), GTK_WIDGET(self->menu_bar), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(box1), GTK_WIDGET(self->toolbar), FALSE, FALSE, 5);
    gtk_box_pack_start(GTK_BOX(box1), GTK_WIDGET(pane1), TRUE, TRUE, 5);

    g_signal_connect(G_OBJECT(self->toolbar->refreshTb), "clicked",
                    G_CALLBACK(update_clicked_cb), self->notebook->active_wifis);
    g_signal_connect(G_OBJECT(self->toolbar->exitTb), "clicked",
                    G_CALLBACK(on_shutdown_cb), self);

    g_signal_connect(G_OBJECT(self->menu_bar->wired_connection), "activate",
                G_CALLBACK(on_wired_connection_cb), self->netaid_config);
    g_signal_connect(G_OBJECT(self->menu_bar->disconnect), "activate",
                G_CALLBACK(on_disconnect_cb), self->netaid_config);
    g_signal_connect(G_OBJECT(self->menu_bar->quit), "activate",
                G_CALLBACK(on_shutdown_cb), self);
    g_signal_connect(G_OBJECT(self->menu_bar->about), "activate",
                G_CALLBACK(on_about_cb), self);

    g_signal_connect(G_OBJECT(self), "notify::ifname",
                G_CALLBACK(on_ifname_property_notify_event), self);

    g_object_bind_property(G_OBJECT(self), "connected",
                       G_OBJECT(self->menu_bar), "connected", 
                       G_BINDING_BIDIRECTIONAL | G_BINDING_SYNC_CREATE);
    g_object_bind_property(G_OBJECT(self), "connected",
                       G_OBJECT(self->toolbar), "connected", 
                       G_BINDING_BIDIRECTIONAL | G_BINDING_SYNC_CREATE);

#if GTK_MAJOR_VERSION == 2

    gtk_paned_set_position(GTK_PANED(pane1), floor(WIN_HEIGHT/2 + 10));

#elif GTK_MAJOR_VERSION == 3

    gtk_paned_set_position(GTK_PANED(pane1), floor(WIN_HEIGHT/2));
    gtk_paned_set_position(GTK_PANED(pane2), floor(WIN_HEIGHT/10));

    _paned_change_handle_size_green(GTK_WIDGET(pane1));

#endif

    term1 = vte_terminal_new();
    gtk_widget_set_sensitive(term1, FALSE);
    _paned_change_handle_size_green(pane1);

    vte_terminal_set_scroll_on_output(VTE_TERMINAL(term1), TRUE);

    gtk_paned_add1(GTK_PANED(pane1), box2);
    gtk_box_pack_start(GTK_BOX(box2), GTK_WIDGET(self->notebook), TRUE, TRUE, 5);
    gtk_paned_add2(GTK_PANED(pane1), pane2);

    gtk_paned_pack1(GTK_PANED(pane2), term1, TRUE, TRUE);

    gtk_widget_set_size_request(GTK_WIDGET(pane1), 15, 15);
    gtk_widget_set_size_request(GTK_WIDGET(term1), 10, 50);

    gtk_window_set_position(GTK_WINDOW(self), GTK_WIN_POS_CENTER);
    gtk_window_set_default_size(GTK_WINDOW(self), WIN_WIDTH, WIN_HEIGHT);
 
    gtk_box_pack_start(GTK_BOX(box1), GTK_WIDGET(self->label), FALSE, FALSE, 5);

    g_str = g_string_new(NULL);

    // lockfile
    {
        const char *user_home;
        if (userinfo_init())
            printf("Cannot retrieve username and home: %s\n", strerror(errno));
        user_home = userhome();
        string_format(g_str, "%s/.simple-netaid-gtk.lock", user_home);
    }

    {

#if GTK_MAJOR_VERSION == 2

        const char *params[] = { "/bin/sh", "-c",
                                self->netaid_config->p.netlink_monitor, (char*)0 };
        const char *env[] = { "PATH=/usr/bin:/bin:./usr/local/bin:/bin", (char*)0 };

        vte_terminal_fork_command(VTE_TERMINAL(term1), "sh", (char**)params,
				(char**)env, "~/", FALSE, FALSE, FALSE);

#elif GTK_MAJOR_VERSION == 3

        g_string_erase(g_str, 0, -1);
        string_format(g_str, "%s", self->netaid_config->p.netlink_monitor);
        puts(g_str->str);
        gchar **envp = NULL;
        gchar **command = NULL;
        g_shell_parse_argv(g_str->str, NULL, &command, NULL);

        vte_terminal_spawn_async(VTE_TERMINAL(term1), VTE_PTY_DEFAULT, NULL, 
				command, NULL, (GSpawnFlags)0, 
				NULL, NULL, NULL, -1, NULL, NULL, NULL);
#endif
    }

    g_string_free(g_str, TRUE);

    g_signal_connect_swapped(VTE_TERMINAL(term1), "child-exited",
                            G_CALLBACK(gtk_window_close), self);

    /*
     * Set the unix signal handling up.
     * First create a pipe.
     */
    if (pipe(signal_pipe)) {
        perror("pipe");
        exit(1);
    }

    /*
     * put the write end of the pipe into nonblocking mode,
     * need to read the flags first, otherwise we would clear other flags too.
     */
    fd_flags = fcntl(signal_pipe[1], F_GETFL);
    if (fd_flags == -1) {
        perror("read descriptor flags");
        exit(1);
    }

    if (fcntl(signal_pipe[1], F_SETFL, fd_flags | O_NONBLOCK) == -1) {
        perror("write descriptor flags");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_sigaction = &sig_handler;
    sa.sa_flags = SA_NODEFER | SA_SIGINFO;
    sigemptyset(&sa.sa_mask);
    sigaction(SIGINT, &sa, 0);
    sigaction(SIGUSR1, &sa, 0);
    sigaction(SIGUSR2, &sa, 0);

    /* convert the reading end of the pipe into a GIOChannel */
    g_signal_in = g_io_channel_unix_new(signal_pipe[0]);

    /*
     * we only read raw binary data from the pipe,
     * therefore clear any encoding on the channel
     */
    g_io_channel_set_encoding(g_signal_in, NULL, &error);
    if (error != NULL){		/* handle potential errors */
        fprintf(stderr, "g_io_channel_set_encoding failed %s\n",
                error->message);
        exit(1);
    }

    /* put the reading end also into non-blocking mode */
    g_io_channel_set_flags(g_signal_in,
            g_io_channel_get_flags(g_signal_in)
            | G_IO_FLAG_NONBLOCK, &error);

    if (error != NULL) {		/* tread errors */
        fprintf(stderr, "g_io_set_flags failed %s\n",
                error->message);
        exit(1);
    }

    /* register the reading end with the event loop */
    g_io_add_watch(g_signal_in, G_IO_IN | G_IO_PRI, deliver_signal, self);
    
    self->timer_handler_id = g_timeout_add(1000, (GSourceFunc)time_handler, (gpointer)self);
}

GtkWidget *netaid_window_new(GtkApplication *app, gboolean systray_ok)
{
    g_return_val_if_fail(GTK_IS_APPLICATION (app), NULL);
    
    NetaidWindow *self = g_object_new(NETAID_TYPE_WINDOW, "application", app, NULL);
    
    NetaidWindowPrivate *priv = netaid_window_get_instance_private(self);
   
    priv->systray_ok = systray_ok;

    if (systray_ok) {

        GdkWindow *root_win;
        GdkWindow *gdk_win;
        
        /* add the status icon */
        self->status_icon = netaid_status_icon_new();
        
        gtk_widget_set_can_focus(GTK_WIDGET(self), TRUE);
        gtk_window_set_accept_focus(GTK_WINDOW(self), TRUE);
        
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(self), TRUE);
        
        g_object_set(G_OBJECT(self), "focus-on-map", TRUE, NULL);
    
        // Enable focus mask
        gtk_widget_add_events(GTK_WIDGET(self), GDK_FOCUS_CHANGE_MASK);

        g_signal_connect(GTK_WIDGET(self), "window-state-event", 
                        G_CALLBACK(on_window_state_changed), NULL);        
        
        gtk_widget_add_events(GTK_WIDGET(self), GDK_VISIBILITY_NOTIFY_MASK);
        g_signal_connect(G_OBJECT(self), "visibility-notify-event",
                        G_CALLBACK(on_visibility_changed), NULL);
                                                
        g_signal_connect(self, "delete-event", G_CALLBACK(on_window_delete_event_cb), self);

        g_signal_connect(self->status_icon->menuItemShow, "activate", 
                        G_CALLBACK(on_toggle_cb), self);
        g_signal_connect(self->status_icon->menuItemExit, "activate",
                        G_CALLBACK(on_shutdown_cb), self);
        g_signal_connect(self->status_icon, "activate", G_CALLBACK(on_toggle_cb), self);

        g_signal_connect(G_OBJECT(self->status_icon), "popup-menu",
                        G_CALLBACK(on_status_icon_popup), self);
    
        //g_signal_connect(G_OBJECT(self->status_icon->menu), "deactivate",
          //              G_CALLBACK(on_menu_closed), self);
    
        gtk_widget_show_all(GTK_WIDGET(self));
    
        self->display = GDK_DISPLAY_XDISPLAY(gdk_display_get_default());
        atom_desktop_x11 = XInternAtom(self->display, "_NET_CURRENT_DESKTOP", False);
        root_win = gdk_get_default_root_window();
        gdk_window_set_events(root_win, GDK_PROPERTY_CHANGE_MASK);
        gdk_window_add_filter(root_win, my_x11_filter, self);
        gdk_win = gtk_widget_get_window(GTK_WIDGET(self));
        self->xid = GDK_WINDOW_XID(gdk_win);
        g_print("The XID of my window is: %lu\n", (unsigned long)self->xid);

        self->iconified = FALSE;
        gtk_window_iconify(GTK_WINDOW(self));

    } else {
        /* regular case */
        g_signal_connect(self, "delete-event", G_CALLBACK(on_window_delete_event_cb), self);
    
        gtk_widget_show_all(GTK_WIDGET(self));
        gtk_window_present(GTK_WINDOW(self));
        gtk_widget_grab_focus(GTK_WIDGET(self));
    }

    return GTK_WIDGET (self);
}

void on_shutdown_cb(GtkWidget *widget, gpointer user_data)
{
    if (user_data == NULL || !GTK_IS_WINDOW(user_data)) {
        g_warning("on_shutdown_cb: user_data is not a valid window");
        gtk_main_quit();
        return;
    }

    GtkWindow *window = GTK_WINDOW(user_data);
    GtkApplication *app = gtk_window_get_application(window);
    if (app)
        g_application_quit(G_APPLICATION(app));
    else
        gtk_main_quit();
}

gboolean on_window_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    g_return_val_if_fail(NETAID_IS_WINDOW(user_data), FALSE);
    
    NetaidWindow *self = NETAID_WINDOW(user_data);

    NetaidWindowPrivate *priv = netaid_window_get_instance_private(self);

    if (priv->systray_ok) {
        gtk_window_iconify(GTK_WINDOW(self));
        return TRUE;
    } else {
        on_shutdown_cb(GTK_WIDGET(self), self);
        return FALSE; 
    }
}

gboolean on_visibility_changed(GtkWidget *widget, GdkEventVisibility *event, gpointer data)
{
    g_return_val_if_fail(NETAID_IS_WINDOW(widget), FALSE);
    
    NetaidWindow *self = NETAID_WINDOW(widget);

    self->iconified = (event->state == GDK_VISIBILITY_FULLY_OBSCURED);

    return FALSE;
}

gboolean on_toggle_cb(GtkWidget *widget, gpointer user_data)
{
    g_return_val_if_fail(NETAID_IS_WINDOW(user_data), FALSE);
    
    NetaidWindow *self = NETAID_WINDOW(user_data);
    
    /* Only read the current state */
    if (self->iconified) {
        /* restore */
        gtk_window_deiconify(GTK_WINDOW(self));
        gtk_window_present(GTK_WINDOW(self));
    } else {
        /* minimize */
        gtk_window_iconify(GTK_WINDOW(self));
    }

    return TRUE;
}

gboolean on_window_state_changed(GtkWidget *widget, GdkEventWindowState *event, gpointer data)
{
    g_return_val_if_fail(NETAID_IS_WINDOW(widget), FALSE);

    NetaidWindow *self = NETAID_WINDOW(widget);

    if (event->changed_mask & GDK_WINDOW_STATE_ICONIFIED)
        self->iconified = (event->new_window_state & GDK_WINDOW_STATE_ICONIFIED);

    return FALSE;
}

void netaid_window_status_icon_update(NetaidWindow *self)
{
    g_return_if_fail(NETAID_IS_WINDOW(self));
    
    if (self->connected) {
        struct sbuf s;
        GdkPixbuf *pixbuf = get_icon_connected();
        sbuf_init(&s);
        //netproc(&s);
        sbuf_concat(&s, 2, "Connected to ", self->ifname->str);
        gtk_status_icon_set_tooltip_text(GTK_STATUS_ICON(self->status_icon), s.buf);
        sbuf_free(&s);
        gtk_status_icon_set_from_pixbuf(GTK_STATUS_ICON(self->status_icon), pixbuf);    
        g_clear_object(&pixbuf);
    } else {
        GdkPixbuf *pixbuf = get_icon_disconnected();
        gtk_status_icon_set_tooltip_text(GTK_STATUS_ICON(self->status_icon), "Disconnected");
        gtk_status_icon_set_from_pixbuf(GTK_STATUS_ICON(self->status_icon), pixbuf);    
        g_clear_object(&pixbuf);
    }
}

