/*
 * netaid-active-wifis.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include "netaid-saved-wifis.h"
#include "netaid-window.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/inotify.h>

#include <globals.h>

enum {
    COLUMN_NETWORK = 0,
    N_COLUMNS,
};

enum {
    PROP_0,
    PROP_FILE,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES];

G_DEFINE_TYPE(NetaidSavedWifis, netaid_saved_wifis, GTK_TYPE_TREE_VIEW)

static void netaid_saved_wifis_finalize(GObject *object)
{
    G_OBJECT_CLASS(netaid_saved_wifis_parent_class)->finalize(object);
}

static void netaid_saved_wifis_constructed(GObject *object)
{
    NetaidSavedWifis *netaid_saved_wifis;

    netaid_saved_wifis = NETAID_SAVED_WIFIS(object);
    G_OBJECT_CLASS(netaid_saved_wifis_parent_class)->constructed(object);
}

static void netaid_saved_wifis_set_property(GObject *object,
        guint property_id, const GValue *value, GParamSpec *pspec)
{
    NetaidSavedWifis *self = NETAID_SAVED_WIFIS(object);

    switch (property_id) {
	case PROP_FILE:
		self->file = g_strdup(g_value_get_string(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    }
}

static void netaid_saved_wifis_get_property(GObject *object, guint property_id,
        GValue *value, GParamSpec *pspec)
{
    NetaidSavedWifis *self = NETAID_SAVED_WIFIS(object);

    switch (property_id) {
    case PROP_FILE:
		g_value_set_string(value, self->file);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    }
}

static void netaid_saved_wifis_class_init(NetaidSavedWifisClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
    GtkContainerClass *container_class = GTK_CONTAINER_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    object_class->constructed = netaid_saved_wifis_constructed;
    object_class->finalize = netaid_saved_wifis_finalize;

    properties[PROP_FILE] = g_param_spec_string("file", NULL, NULL, NULL,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

    object_class->set_property = netaid_saved_wifis_set_property;
    object_class->get_property = netaid_saved_wifis_get_property;

    g_object_class_install_property(object_class, PROP_FILE, properties[PROP_FILE]);
}

static void on_press_event_cb(GtkTreeView *view, GtkTreePath *path, 
		GtkTreeViewColumn *col, gpointer user_data)
{
    NetaidSavedWifis *self = NETAID_SAVED_WIFIS(view);
    gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(self), FALSE);

    GtkTreeModel *model;
    GtkTreeIter   iter;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(self));

    if (gtk_tree_model_get_iter(model, &iter, path))
    {
        gchar *name;
        GString *g_str = g_string_new("");
        gtk_tree_model_get(model, &iter, COLUMN_NETWORK, &name, -1);
		g_string_printf(g_str, "%s/%s", (const char *)wifi_dir, name);
        g_object_set(G_OBJECT(self), "file", g_str->str, NULL);
        g_string_free(g_str, TRUE);
        g_free(name);
    }
}

static void netaid_saved_wifis_init(NetaidSavedWifis *self)
{
    GtkTreeViewColumn *col;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    DIR *dirp;
    struct dirent *E;
    FILE *fstab = NULL;
    static const GtkDialogFlags INOTIFY_FLAGS = IN_CREATE | IN_DELETE | IN_MOVED_TO | IN_MOVED_FROM;

    GtkListStore *store = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, -1);

    gtk_tree_view_set_model(GTK_TREE_VIEW(self), GTK_TREE_MODEL(store));

    /* destroy store automatically with view */
    g_object_unref(store);

    gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(self), TRUE);

    gtk_widget_set_vexpand(GTK_WIDGET(self), TRUE);
    gtk_widget_set_hexpand(GTK_WIDGET(self), TRUE);

    renderer = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(col, renderer, TRUE);
    gtk_tree_view_column_add_attribute(col, renderer, "text", COLUMN_NETWORK);
    gtk_tree_view_column_set_title(col, " NETWORK");
    gtk_tree_view_append_column(GTK_TREE_VIEW(self),col);

    gtk_tree_view_set_activate_on_single_click(GTK_TREE_VIEW(self), TRUE);

    g_signal_connect(self, "row-activated", G_CALLBACK(on_press_event_cb), NULL);

    dirp = opendir((const char *)wifi_dir);
    if (!dirp) {
        fprintf(stderr, "error opening current directory: %s\n", strerror(errno));
        exit(EXIT_FAILURE);
    };

    while ((errno = 0, E = readdir(dirp))) {
        struct stat sb;
        if (lstat(E->d_name, &sb)) {
            if (sb.st_mode & S_IFMT) {
                gtk_list_store_append(GTK_LIST_STORE(store), &iter);
                gtk_list_store_set(GTK_LIST_STORE(store), &iter, COLUMN_NETWORK, E->d_name, -1);
            }
        }
    }

    /*  Initialize inotify & signalfd  and open /proc/self/mountinfo  */
    self->inotfd = inotify_init1(IN_CLOEXEC);
    if (self->inotfd < 0) {
        fprintf(stderr, "cannot initialize inotify: %s\n", strerror(errno));
        // TODO: handle error
        exit(EXIT_FAILURE);
    }

    if (inotify_add_watch(self->inotfd, (const char *)wifi_dir, INOTIFY_FLAGS) == -1) {
        fprintf(stderr, "cannot add inotify watch: %s\n", strerror(errno));
        // TODO: handle error
        exit(EXIT_FAILURE);
    }

    self->ginotfd = g_io_channel_unix_new(self->inotfd);

    /* Declare input watch */
    g_io_add_watch(self->ginotfd, G_IO_IN, callback_inotify, self);

    gtk_widget_show_all(GTK_WIDGET(self));
}

NetaidSavedWifis *netaid_saved_wifis_new()
{
    NetaidSavedWifis *saved_wifis;

    saved_wifis = g_object_new(NETAID_TYPE_SAVED_WIFIS, NULL);

    return NETAID_SAVED_WIFIS(saved_wifis);
}

gboolean callback_inotify(GIOChannel *source, GIOCondition condition,
		gpointer user_data)
{
	NetaidSavedWifis *saved_wifis = user_data;
	int inotfd;
	int len;
	char *b;

	inotfd =(int)g_io_channel_unix_get_fd(source);
    struct inot_event {
        int wd;
        uint32_t mask;
        uint32_t cookie;
        uint32_t len;
        char name[NAME_MAX + 1];
    };

    char buf[sizeof(struct inot_event)];
    const size_t header_size = sizeof(int) + sizeof(uint32_t) * 3;

    /* read inotify event(s) */
    len = read(inotfd, &buf, sizeof(struct inot_event));
    if (len >= header_size) { /* we've read at least one event */
        struct inot_event *inot_evt;

        /* We now have at least one inotify event in the buffer */
        for (b = buf, inot_evt =(struct inot_event *)b; 
					len >= header_size;
					b +=(header_size + inot_evt->len), 
					len -=(header_size + inot_evt->len),
					inot_evt =(struct inot_event *)b) {

            if (inot_evt->mask & IN_DELETE) {
				GtkTreeModel *model;
				GtkTreeIter iter;
				int column_index = 0; 

				g_print("IN_DELETE: %s\n", inot_evt->name);
				model = gtk_tree_view_get_model(GTK_TREE_VIEW(saved_wifis));
				
				// Get the first iterator in the list
				if (gtk_tree_model_get_iter_first(model, &iter)) {
					gboolean valid = TRUE;
					while (valid) {
						g_autofree char *current_value = NULL;
            
						// Retrieve value from the specified column
						gtk_tree_model_get(model, &iter, column_index, 
								&current_value, -1);

						if (g_strcmp0(current_value, inot_evt->name) == 0) {
							// Remove the row. gtk_list_store_remove sets 'iter'
							// to the next valid row, or invalidates it if it was 
							// the last row.
							valid = gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
							// Stop after finding the first match
							break; // Ya lo encontramos y borramos, salimos del bucle
						} else {
							// Continue iterating only if we didn't find the item yet
							valid = gtk_tree_model_iter_next(model, &iter);
						}
					}
				}
            } else if (inot_evt->mask & IN_MOVED_FROM) {
				GtkTreeModel *model;
				GtkTreeIter iter;
				int column_index = 0; 

				g_print("IN_MOVED_FROM: %s\n", inot_evt->name);
				model = gtk_tree_view_get_model(GTK_TREE_VIEW(saved_wifis));
				
				// Get the first iterator in the list
				if (gtk_tree_model_get_iter_first(model, &iter)) {
					gboolean valid = TRUE;
					while (valid) {
						g_autofree char *current_value = NULL;
            
						// Retrieve value from the specified column
						gtk_tree_model_get(model, &iter, column_index, 
								&current_value, -1);

						if (g_strcmp0(current_value, inot_evt->name) == 0) {
							// Remove the row. gtk_list_store_remove sets 'iter'
							// to the next valid row, or invalidates it if it was 
							// the last row.
							valid = gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
							// Stop after finding the first match
							break; // Ya lo encontramos y borramos, salimos del bucle
						} else {
							// Continue iterating only if we didn't find the item yet
							valid = gtk_tree_model_iter_next(model, &iter);
						}
					}
				}
            } else if (inot_evt->mask & IN_MOVED_TO) {
				GtkTreeModel *model;
				GtkTreeIter iter;

                g_print("IN_MOVED_TO: %s\n", inot_evt->name);

				model = gtk_tree_view_get_model(GTK_TREE_VIEW(saved_wifis));

				gtk_list_store_append(GTK_LIST_STORE(model), &iter);
				gtk_list_store_set(GTK_LIST_STORE(model),
						&iter, COLUMN_NETWORK, inot_evt->name, -1);

            } else if (inot_evt->mask & IN_CREATE) {
				GtkTreeModel *model;
				GtkTreeIter iter;
				GtkTreePath *path;
				GtkTreeRowReference *ref;
                
                g_print("ADDED: %s\n", inot_evt->name);

				model = gtk_tree_view_get_model(GTK_TREE_VIEW(saved_wifis));

				gtk_list_store_append(GTK_LIST_STORE(model), &iter);
				gtk_list_store_set(GTK_LIST_STORE(model), 
						&iter, COLUMN_NETWORK, inot_evt->name, -1);
					
				path = gtk_tree_model_get_path(GTK_TREE_MODEL(model), &iter);
				ref = gtk_tree_row_reference_new(GTK_TREE_MODEL(model), path);
				gtk_tree_path_free(path);
            }
        }
    } else {
        fprintf(stderr, "error reading inotify event: %s\n", strerror(errno));
    }
  
	return TRUE;
}

