/*
 * netaid-password-dialog.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include "netaid-password-dialog.h"
#include "netaid-window.h"
#include "netaid-config.h"
#include <globals.h>

#include <simple-netaid/iproute.h>

enum {
    PROP_0,
    PROP_ESSID,
    PROP_KEY,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = {NULL, };

G_DEFINE_TYPE(NetaidPasswordDialog, netaid_password_dialog, GTK_TYPE_DIALOG)

static void netaid_password_dialog_set_property(GObject *object, 
		guint property_id, const GValue *value, GParamSpec *pspec)
{
    NetaidPasswordDialog *self = NETAID_PASSWORD_DIALOG(object);

    if (property_id == PROP_ESSID) {
        self->essid = g_strdup(g_value_get_string(value));
    } else if (property_id == PROP_KEY) {
		gchar *label;
        self->key = g_value_get_boolean(value);
        if (self->key) {
			label = g_strconcat("  Enter the password for \"", self->essid, "\":  ", NULL);
			gtk_widget_set_sensitive(GTK_WIDGET(self->connect_button), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(self->connect_and_save_button), FALSE);
			gtk_widget_set_visible(GTK_WIDGET(self->entry), TRUE);    
			gtk_widget_set_visible(GTK_WIDGET(self->check_button), TRUE);
			g_signal_connect(GTK_EDITABLE(self->entry), "notify::text", 
					G_CALLBACK(update_button_cb), self);
		} else {
			label = g_strconcat(
				"The network \"", self->essid, "\" is open\n"
				"and doesn't require a password", 
				NULL
			);
		}
		gtk_label_set_text(GTK_LABEL(self->label), label);
		g_free(label);
    } else {
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    }
}

static void netaid_password_dialog_get_property(GObject *object, 
		guint property_id, GValue *value, GParamSpec *pspec)
{
    NetaidPasswordDialog *self = NETAID_PASSWORD_DIALOG(object);

    if (property_id == PROP_ESSID)
        g_value_set_string(value, g_strdup(self->essid));
    else if (property_id == PROP_KEY)
        g_value_set_boolean(value, self->key);
    else
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void netaid_password_dialog_finalize(GObject *object)
{
    NetaidPasswordDialog *self = NETAID_PASSWORD_DIALOG(object);

    if (self->essid) {
        g_free(self->essid);
        self->essid = NULL;
    }

    G_OBJECT_CLASS(netaid_password_dialog_parent_class)->finalize(object);
}

static void netaid_password_dialog_constructed(GObject *object)
{
    G_OBJECT_CLASS(netaid_password_dialog_parent_class)->constructed(object);
}

static void netaid_password_dialog_class_init(NetaidPasswordDialogClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
    GtkContainerClass *container_class = GTK_CONTAINER_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    object_class->constructed = netaid_password_dialog_constructed;
    object_class->finalize = netaid_password_dialog_finalize;

    object_class->set_property = netaid_password_dialog_set_property;
    object_class->get_property = netaid_password_dialog_get_property;

    properties[PROP_ESSID] = g_param_spec_string("essid", 
			"string", "string", "", G_PARAM_READWRITE);

    properties[PROP_KEY] = g_param_spec_boolean("key", NULL, NULL,
            TRUE,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS /* | G_PARAM_EXPLICIT_NOTIFY */);

    g_object_class_install_properties(object_class, N_PROPERTIES, properties);
}

static void netaid_password_dialog_init(NetaidPasswordDialog *self)
{
    GtkWidget *box;

    gtk_window_set_default_size(GTK_WINDOW(self), 320, -1);

    box = gtk_dialog_get_content_area(GTK_DIALOG(self));

    self->label = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(box), self->label, TRUE, TRUE, 0);
    
    self->entry = gtk_entry_new();
    g_object_set(self->entry, "placeholder-text", "Password",
        //"activates-default", TRUE,
        NULL
	);

	gtk_entry_set_visibility(GTK_ENTRY(self->entry), FALSE);
    gtk_box_pack_start(GTK_BOX(box), self->entry, TRUE, TRUE, 0);

    self->check_button = gtk_check_button_new_with_label("Show password");
    g_signal_connect(self->check_button, "toggled", 
			G_CALLBACK(check_button_pressed_cb), self);
    gtk_widget_set_sensitive(GTK_WIDGET(self->check_button), TRUE);
    gtk_box_pack_start(GTK_BOX(box), self->check_button, TRUE, TRUE, 0);
    self->connect_button = gtk_button_new_with_mnemonic("Connect");
    g_signal_connect(self->connect_button, "clicked", 
			G_CALLBACK(connect_button_pressed_cb), self);
    gtk_box_pack_start(GTK_BOX(box), self->connect_button, TRUE, TRUE, 0);

    self->connect_and_save_button = gtk_button_new_with_mnemonic("Connect + Save");
	g_signal_connect(self->connect_and_save_button, "clicked", 
			G_CALLBACK(connect_and_save_button_pressed_cb), self);
    gtk_box_pack_start(GTK_BOX(box), self->connect_and_save_button, TRUE, TRUE, 0);

    self->cancel_button = gtk_button_new_with_mnemonic("Cancel");
    g_signal_connect_swapped(self->cancel_button, "clicked", 
			G_CALLBACK(gtk_widget_destroy), self);
    gtk_widget_set_sensitive(GTK_WIDGET(self->cancel_button), TRUE);
    gtk_box_pack_start(GTK_BOX(box), self->cancel_button, TRUE, TRUE, 0);
    gtk_widget_set_margin_top(GTK_WIDGET(self->label), 10);
    gtk_widget_set_margin_top(GTK_WIDGET(self->check_button), 10);
    gtk_widget_set_margin_top(GTK_WIDGET(self->connect_button), 10);
    gtk_widget_set_margin_right(GTK_WIDGET(self->connect_button), 5);
    gtk_widget_set_margin_left(GTK_WIDGET(self->connect_button), 5);
    gtk_widget_set_margin_top(GTK_WIDGET(self->connect_and_save_button), 10);
    gtk_widget_set_margin_right(GTK_WIDGET(self->connect_and_save_button), 5);
    gtk_widget_set_margin_left(GTK_WIDGET(self->connect_and_save_button), 5);
    gtk_widget_set_margin_top(GTK_WIDGET(self->cancel_button), 10);
    gtk_widget_set_margin_right(GTK_WIDGET(self->cancel_button), 5);
    gtk_widget_set_margin_left(GTK_WIDGET(self->cancel_button), 5);
    gtk_widget_set_margin_bottom(GTK_WIDGET(self->cancel_button), 5);
     
    gtk_widget_set_visible(GTK_WIDGET(self->entry), FALSE);
    gtk_widget_set_no_show_all(GTK_WIDGET(self->entry), TRUE); 
    
    gtk_widget_set_visible(GTK_WIDGET(self->check_button), FALSE);
    gtk_widget_set_no_show_all(GTK_WIDGET(self->check_button), TRUE);
}

NetaidPasswordDialog *netaid_password_dialog_new(GtkWindow *window, const gchar *str, gboolean key)
{
    GtkWidget *password_dialog;

    password_dialog = g_object_new(NETAID_TYPE_PASSWORD_DIALOG,
        "title", "Password dialog",
        "transient-for", window,
        "essid", str,
        "key", key,
        NULL
	);

    return NETAID_PASSWORD_DIALOG(password_dialog);
}

void update_button_cb(GObject *object, GParamSpec *pspec, gpointer user_data)
{
    NetaidPasswordDialog *self = user_data;
    GtkEntry *entry = GTK_ENTRY(object);
    const char *text = gtk_entry_get_text(GTK_ENTRY(entry));
    gtk_widget_set_sensitive(self->connect_button, text[0] != '\0');
    gtk_widget_set_sensitive(self->connect_and_save_button, text[0] != '\0');
}

void check_button_pressed_cb(GtkButton *widget, gpointer user_data)
{
    NetaidPasswordDialog *self = user_data;
    GtkCheckButton *check_button = GTK_CHECK_BUTTON(widget);
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check_button)))
        gtk_entry_set_visibility(GTK_ENTRY(self->entry), TRUE);
    else
        gtk_entry_set_visibility(GTK_ENTRY(self->entry), FALSE);
}

void connect_button_pressed_cb(GtkWidget *widget, gpointer user_data)
{
    NetaidPasswordDialog *self =(NetaidPasswordDialog *)user_data;
    GtkWindow *win = gtk_window_get_transient_for(GTK_WINDOW(self));
 
    if (GTK_IS_WINDOW(win)) {
		NetaidWindow *w = NETAID_WINDOW(win);
		const char *password = gtk_entry_get_text(GTK_ENTRY(self->entry));
		netaid_config_spawn_async_command(NETAID_CONFIG(w->netaid_config), 
				w->netaid_config->p.wireless_connection,
				self->essid, password, NULL);
	}
}

void connect_and_save_button_pressed_cb(GtkWidget *widget, gpointer user_data)
{
    NetaidPasswordDialog *self =(NetaidPasswordDialog *)user_data;
    GtkWindow *win = gtk_window_get_transient_for(GTK_WINDOW(self));
	const char *ifname = iproute();

	if (*ifname != 0) {	
		GtkWidget *dialog = gtk_message_dialog_new (NULL,
				GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
				"\nAlready connected to %s\n", ifname);
       gtk_window_set_title(GTK_WINDOW(dialog), "simple-netaid info message");
       gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
       gtk_dialog_run(GTK_DIALOG(dialog));
       gtk_widget_destroy(dialog);
       return;
    }

    if (GTK_IS_WINDOW(win)) {
		/* Required for w->netaid_config */
		NetaidWindow *w = NETAID_WINDOW(win);
		
		/* Not required because buttons are inhibited while the GtkEntry is empty */
		/*
		if (!password && self->key) {
			GtkWidget *dialog = gtk_message_dialog_new (NULL, 
					GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
					"\n%s is encrypted and requires password\n", self->essid);
			gtk_window_set_title(GTK_WINDOW(dialog), "simple-netaid info message");
			gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);
		}
		*/

		const char *info_message = "Enter the name for the wpa config file:";			
		GtkWidget *dialog = gtk_message_dialog_new (NULL,
				GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
				"%s", info_message);
		
		gtk_window_set_title(GTK_WINDOW(dialog), "Simple Netaid dialog");
		gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 150);

		GtkWidget *entry = gtk_entry_new();
		gtk_entry_set_visibility(GTK_ENTRY(entry), TRUE);

		GtkWidget *content_area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
		gtk_box_pack_start(GTK_BOX(content_area), entry, FALSE, FALSE, 5);

		/* Dialog children aren't always shown by default */
		gtk_widget_show_all(dialog);
		gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);

		gint response = gtk_dialog_run(GTK_DIALOG(dialog));
		if (response == GTK_RESPONSE_OK) {
			const gchar *filename = gtk_entry_get_text(GTK_ENTRY(entry));
			const gchar *password = gtk_entry_get_text(GTK_ENTRY(self->entry));
			if (filename && strlen(filename) > 0) {
				GString *g_str = g_string_new("");
				g_string_printf(g_str, "%s/%s", (const char *)wifi_dir, filename);
				netaid_config_spawn_async_command(NETAID_CONFIG(w->netaid_config), 
						w->netaid_config->p.wireless_connection, self->essid, password, g_str->str);
				g_string_free(g_str, TRUE);
			}
		}

		gtk_widget_destroy(dialog);
		dialog = NULL;
	}
}
