/*
 * netaid-menu_bar.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include "netaid-menu-bar.h"
#include "netaid-window.h"
#include "netaid-config.h"

#include <simple-netaid/sbuf.h>
#include <simple-netaid/netproc.h>
#include <simple-netaid/iproute.h>

/* Functions defined in set_icon.c */
GdkPixbuf *get_brand_icon();
GdkPixbuf *get_icon_wired();
GdkPixbuf *get_icon_connected();
GdkPixbuf *get_icon_disconnected();

enum {
    PROP_0,
    PROP_CONNECTED,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = {NULL, };

G_DEFINE_TYPE(NetaidMenuBar, netaid_menu_bar, GTK_TYPE_MENU_BAR)

// Signal handler for the "property-notify-event"
static gboolean on_property_notify_event(GObject *gobject, 
		GParamSpec *pspec, gpointer user_data)
{
    NetaidMenuBar *menu_bar =(NetaidMenuBar *)gobject;
    const char *name;
    gboolean is_connected;
    name = g_param_spec_get_name(pspec);
    g_object_get(G_OBJECT(menu_bar), name, &is_connected, NULL);
    if (is_connected) {
        gtk_widget_set_sensitive(menu_bar->wired_connection, FALSE);
        gtk_widget_set_sensitive(menu_bar->disconnect, TRUE);
    }
    else {
        gtk_widget_set_sensitive(menu_bar->wired_connection, TRUE);
        gtk_widget_set_sensitive(menu_bar->disconnect, FALSE);
    }
}

static void netaid_menu_bar_set_property(GObject *object, guint property_id, 
		const GValue *value, GParamSpec *pspec)
{
    NetaidMenuBar *self = NETAID_MENU_BAR(object);

    if (property_id == PROP_CONNECTED)
        self->connected = g_value_get_boolean(value);
    else
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void netaid_menu_bar_get_property(GObject *object, guint property_id, 
		GValue *value, GParamSpec *pspec)
{
    NetaidMenuBar *self = NETAID_MENU_BAR(object);

    if (property_id == PROP_CONNECTED)
        g_value_set_boolean(value, self->connected);
    else
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
}

static void netaid_menu_bar_finalize(GObject *object)
{
    G_OBJECT_CLASS(netaid_menu_bar_parent_class)->finalize(object);
}

static void netaid_menu_bar_constructed(GObject *object)
{
    G_OBJECT_CLASS(netaid_menu_bar_parent_class)->constructed(object);
}

static void netaid_menu_bar_class_init(NetaidMenuBarClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
    GtkMenuBarClass *menu_bar_class = GTK_MENU_BAR_CLASS(klass);
    GtkContainerClass *container_class = GTK_CONTAINER_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    object_class->constructed = netaid_menu_bar_constructed;
    object_class->finalize = netaid_menu_bar_finalize;
    object_class->set_property = netaid_menu_bar_set_property;
    object_class->get_property = netaid_menu_bar_get_property;

    properties[PROP_CONNECTED] = g_param_spec_boolean("connected", NULL, NULL,
            TRUE,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS /* | G_PARAM_EXPLICIT_NOTIFY */);

    g_object_class_install_properties(object_class, N_PROPERTIES, properties);
}

static void netaid_menu_bar_init(NetaidMenuBar *self)
{
    GtkWidget *menuItem1;
    GtkWidget *menuItem2;
    GtkWidget *submenu1;
    GtkWidget *submenu2;

    GtkWidget *image;
    menuItem1 = gtk_menu_item_new_with_mnemonic("Menu");
    submenu1 = gtk_menu_new();

    self->wired_connection =  gtk_image_menu_item_new_with_mnemonic("Wired connection");
    image = gtk_image_new_from_pixbuf(get_icon_wired());
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(self->wired_connection), image);

    self->disconnect =  gtk_image_menu_item_new_from_stock(GTK_STOCK_DISCONNECT, NULL);

    self->quit = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);

    /* packing */
    gtk_menu_shell_append(GTK_MENU_SHELL(submenu1), self->wired_connection);
    gtk_menu_shell_append(GTK_MENU_SHELL(submenu1), self->disconnect);
    gtk_menu_shell_append(GTK_MENU_SHELL(submenu1), self->quit);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuItem1), submenu1);
    gtk_menu_shell_append(GTK_MENU_SHELL(self), menuItem1);
    
    menuItem2 = gtk_menu_item_new_with_mnemonic("Help");
    submenu2 = gtk_menu_new();
    
    self->about =  gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(submenu2), self->about);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuItem2), submenu2);
    gtk_menu_shell_append(GTK_MENU_SHELL(self), menuItem2);

    netaid_menu_bar_update(self);

    g_signal_connect(G_OBJECT(self), "notify::connected", 
			G_CALLBACK(on_property_notify_event), NULL);

    gtk_widget_show_all(GTK_WIDGET(self));
}

NetaidMenuBar *netaid_menu_bar_new()
{
    NetaidMenuBar *menu_bar;

    menu_bar = g_object_new(NETAID_TYPE_MENU_BAR, NULL);

    return NETAID_MENU_BAR(menu_bar);
}

void netaid_menu_bar_update(NetaidMenuBar *menu_bar)
{

}

void on_wired_connection_cb(GtkWidget *widget, gpointer user_data)
{
    NetaidConfig *config =(NetaidConfig *)user_data;
    netaid_config_spawn_async_command(NETAID_CONFIG(config), 
			config->p.wired_connection, NULL, NULL, NULL);
}

void on_disconnect_cb(GtkWidget *widget, gpointer user_data)
{
    NetaidConfig *config =(NetaidConfig *)user_data;
    netaid_config_spawn_async_command(NETAID_CONFIG(config), 
			config->p.disconnect, NULL, NULL, NULL);
}

static void on_dialog_destroy_unblock(GtkWidget *about, gpointer user_data) {

    NetaidWindow *self = NETAID_WINDOW(user_data);
    
    // FORZAR RE-SINCRONIZACIÓN CON EL WM
    gtk_window_present_with_time(GTK_WINDOW(self), gtk_get_current_event_time());
    
    // Esto obliga al WM a re-evaluar la ventana aunque no esté en la taskbar
    gtk_window_set_skip_taskbar_hint(GTK_WINDOW(self), FALSE);
    while (gtk_events_pending())
        gtk_main_iteration();
    
    gtk_window_set_skip_taskbar_hint(GTK_WINDOW(self), TRUE);
}

void on_about_cb(GtkWidget *widget, gpointer user_data)
{
    g_return_if_fail(NETAID_IS_WINDOW(user_data));

    NetaidWindow *parent = NETAID_WINDOW(user_data);
        
    gtk_window_set_skip_taskbar_hint(GTK_WINDOW(parent), FALSE);
    
    GtkWidget *about = gtk_about_dialog_new();
    
    g_signal_connect(about, "response", G_CALLBACK(gtk_widget_destroy), NULL);
    g_signal_connect(about, "destroy", G_CALLBACK(on_dialog_destroy_unblock), parent);
    
    GdkPixbuf *logo = get_brand_icon();

    const gchar *authors[] = {
        "- Aitor CZ (Gtk versions - 2026)",
        "- Edward Bartolo (previous version in C and Lazarus Free Pascal - 2015)",
        NULL
    };

    const gchar *documenters[] = {
        "- Aitor CZ (Gtk versions - 2026)",
        NULL
    };

    // Configure properties:
    g_object_set(G_OBJECT(about),
        "transient-for", GTK_WINDOW(parent),
        "program-name", "Simple Netaid",
        "version", "1.0.8",
        "copyright", "© 2026 Aitor CZ <aitor@genuen.org>",
        "license-type", GTK_LICENSE_GPL_3_0, // Uses built-in license text
        "website", "https://www.genuen.com",
        "authors", authors,
        "documenters", documenters,
        "logo", logo,
        NULL
    );
    
    g_clear_object(&logo);
    
    gtk_widget_show_all(about);
}
