/*
 * netaid-config.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include "netaid-config.h"

#include <stdio.h>
#include <ini.h>

G_DEFINE_TYPE(NetaidConfig, netaid_config, G_TYPE_OBJECT)

static int config_ini_parser(void *config, const char *section, const char *name, const char *value)
{
    Config *p = (Config *)config;

    #define MATCH(s, n) strcmp(section, s) == 0 && strcmp(name, n) == 0

    if (MATCH("devices", "wired_device"))
        p->wired_device = g_strdup(value);
    else if (MATCH("devices", "wireless_device"))
        p->wireless_device = g_strdup(value);
    else if (MATCH("commands", "wired_on"))
        p->wired_on = g_strdup(value);
    else if (MATCH("commands", "wired_off"))
        p->wired_off = g_strdup(value);
    else if (MATCH("commands", "wireless_on"))
        p->wireless_on = g_strdup(value);
    else if (MATCH("commands", "wireless_off"))
        p->wireless_off = g_strdup(value);
    else if (MATCH("commands", "wired_connection"))
        p->wired_connection = g_strdup(value);
    else if (MATCH("commands", "wireless_connection"))
        p->wireless_connection = g_strdup(value);
    else if (MATCH("commands", "disconnect"))
        p->disconnect = g_strdup(value);
    else if (MATCH("commands", "iwlist_scanning"))
        p->iwlist_scanning = g_strdup(value);
    else if (MATCH("commands", "uninstall"))
        p->uninstall = g_strdup(value);
    else if (MATCH("commands", "netlink_monitor"))
        p->netlink_monitor = g_strdup(value);
    else if (MATCH("paths", "path_to_icons"))
        p->path_to_icons = g_strdup(value);
    else
        return 0;  /* unknown section/name, error */

    return 1;
}

static void netaid_config_finalize(GObject *object)
{
    NetaidConfig *netaid_config = NETAID_CONFIG(object);

    if (&netaid_config->p)
        netaid_config_free(&netaid_config->p);

    G_OBJECT_CLASS(netaid_config_parent_class)->finalize(object);
}

static void netaid_config_constructed(GObject *object)
{
    NetaidConfig *netaid_config;

    netaid_config = NETAID_CONFIG(object);
    G_OBJECT_CLASS(netaid_config_parent_class)->constructed(object);
}

static void netaid_config_class_init(NetaidConfigClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->constructed = netaid_config_constructed;
    gobject_class->finalize = netaid_config_finalize;
}

static void netaid_config_init(NetaidConfig *self)
{
    memset(&self->p, 0, sizeof(Config));

    if (ini_parse("/etc/simple-netaid/gtk.ini", config_ini_parser, &self->p) < 0) {
        printf("Can't load 'gtk.ini'\n");
        // TODO: handle error
        exit(EXIT_FAILURE);
    }
}

NetaidConfig* netaid_config_new()
{
    NetaidConfig *netaid_config;

    netaid_config = g_object_new(NETAID_TYPE_CONFIG, NULL);
    return NETAID_CONFIG(netaid_config);
}

// free a config
// always succeeds
int netaid_config_free(Config *p)
{
    if (!p)
        return -1;

    g_clear_pointer(&p->wired_device, g_free);
    g_clear_pointer(&p->wireless_device, g_free);
    g_clear_pointer(&p->wired_on, g_free);
    g_clear_pointer(&p->wired_off, g_free);
    g_clear_pointer(&p->wireless_on, g_free);
    g_clear_pointer(&p->wireless_off, g_free);
    g_clear_pointer(&p->wired_connection, g_free);
    g_clear_pointer(&p->wireless_connection, g_free);
    g_clear_pointer(&p->disconnect, g_free);
    g_clear_pointer(&p->iwlist_scanning, g_free);
    g_clear_pointer(&p->uninstall, g_free);
    g_clear_pointer(&p->netlink_monitor, g_free);
    g_clear_pointer(&p->path_to_icons, g_free);

    return 0;
}

void netaid_config_parse_command(NetaidConfig *self, GString *g_str, 
        const char *essid, const char *password, const char *filename)
{
    // 2. Usar macros o ternarios para asegurar que no pasamos basura
    // Si wireless_device contiene basura, g_string_replace fallará.
    g_string_replace(g_str, "%e", self->p.wired_device ? self->p.wired_device : "", 0);
    g_string_replace(g_str, "%w", self->p.wireless_device ? self->p.wireless_device : "", 0);
    g_string_replace(g_str, "%s", essid ? essid : "", 0);
    g_string_replace(g_str, "%p", password ? password : "", 0);
    g_string_replace(g_str, "%f", filename ? filename : "", 0);
}


// The callback function that is called when data is available on the channel
// Return TRUE to keep the watch active
// Return false to remove the watch
static gboolean on_stdout_cb(GIOChannel *channel, GIOCondition condition, gpointer user_data)
{
    GMainLoop *loop =(GMainLoop *)user_data;

    GError *err = NULL;
    GIOStatus status;
    gchar *line;
    gsize length;

    if (condition == G_IO_IN) {
        // Read a line from the I/O channel
        status = g_io_channel_read_line(channel, &line, &length, NULL, &err);

        if (status == G_IO_STATUS_NORMAL) {
            // Process the line of output
            g_print("Child stdout: %s", line);
            g_free(line);
            return TRUE;
        } else if (status == G_IO_STATUS_EOF) {
            // Child process closed the pipe(finished execution)
            // It would be weird to arrive at this point because,
            // if the pipe has been closed, 'condition' should be
            // equal to G_IO_HUP
            g_print("Child process finished, closing channel.\n");
            g_io_channel_unref(channel);
            return FALSE;
        } else if (status == G_IO_STATUS_ERROR) {
            g_print("Error reading channel: %s\n", err->message);
            g_error_free(err);
            g_io_channel_unref(channel);
            return FALSE;
        }
    } else if (condition == G_IO_HUP) {
        g_io_channel_unref(channel);
        return FALSE;
    }

    return TRUE; /* Keep the watch active for other conditions if necessary */
}

// Callback function to handle the standard error from the child process
static gboolean on_stderr_cb(GIOChannel *channel, GIOCondition condition, gpointer data)
{
    gchar *line = NULL;
    GError *err = NULL;
    gsize len;
    g_print("Child process stderr:\n");
    while (g_io_channel_read_line(channel, &line, &len, NULL, &err) == G_IO_STATUS_NORMAL) {
        g_print("%s", line);
        g_free(line);
        line = NULL;
    }

    return TRUE; /* Continue watching */
}

/*
 * Callback function called when the child process exits.
 */
static void child_watch_cb(GPid child_pid, gint status, gpointer user_data)
{
    GMainLoop *loop =(GMainLoop *)user_data;

    g_print("Child process %d exited with status %d\n", child_pid, status);

    // On some platforms(like Windows), the GPid must be explicitly closed
    // in the callback to free resources.
    g_spawn_close_pid(child_pid);

    // Quit the main loop, as the monitored child process has finished.
    // Optionally quit the main loop if this was the only thing running,
    // as the monitored child process has finished.
    if (loop != NULL)
        g_main_loop_quit(loop);
}

static int spawn_async_command(const char *command)
{
    GError *err = NULL;
    gchar *argv_spawn[] = {"/bin/sh", "-c",(gchar *)command,(gchar *)0};
    GPid child_pid;
    int stdout_pipe, stderr_pipe;
    GIOChannel *stdout_channel, *stderr_channel;
    GMainLoop *loop;

    // Create a new GLib main loop
    loop = g_main_loop_new(NULL, FALSE);

    /* Spawn the child process and capture its stdout */
    if (!g_spawn_async_with_pipes(
            NULL,               /* current working directory */
            argv_spawn,         /* argument list */
            NULL,               /* environment */
            G_SPAWN_DO_NOT_REAP_CHILD, /* flags */
            NULL,               /* child setup function */
            NULL,               /* user data for child setup */
            &child_pid,         /* child process ID(out) */
            NULL,               /* stdin pipe(out) */
            &stdout_pipe,       /* stdout pipe(out) */
            NULL,               /* stderr pipe(out) */
            &err)) {
        g_print("Spawn failed: %s\n", err->message);
        g_error_free(err);
        return 1;
    }

    // Create a GIOChannel from the stdout file descriptor
    stdout_channel = g_io_channel_unix_new(stdout_pipe);

    // Ensure the channel is non-blocking(g_spawn_async_with_pipes does this)
    g_io_channel_set_flags(stdout_channel, G_IO_FLAG_NONBLOCK, NULL);

    // Add a watch to the channel for G_IO_IN condition(data available to read)
    g_io_add_watch(stdout_channel, G_IO_IN | G_IO_HUP, on_stdout_cb, loop);

    // Add a watch for when the child process terminates
    g_child_watch_add(child_pid,(GChildWatchFunc)child_watch_cb, loop);

    g_print("Program started, entering main loop...\n");

    // Run the main loop, which will block until the child process finishes
    g_main_loop_run(loop);

    g_print("Main loop quit. Exiting.\n");

    // Clean up
    g_main_loop_unref(loop);
    g_spawn_close_pid(child_pid);

    return 0;
}

void netaid_config_spawn_async_command(NetaidConfig *self, const char *command,
		const char *essid, const char *password, const char *filename)
{
    GString *g_str = g_string_new(command);
    netaid_config_parse_command(self, g_str, essid, password, filename);
    printf("\n\n%s\n\n", g_str->str);
    spawn_async_command(g_str->str);
    g_string_free(g_str, TRUE);
}
