/*
 * netaid-active-wifis.c
 * Copyright(C) 2026 Aitor Cuadrado Zubizarreta <aitor@genuen.org>
 *
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the COPYING file. *
 */

#include "netaid-active-wifis.h"
#include "netaid-window.h"
#include "netaid-notebook.h"
#include "netaid-toolbar.h"
#include "netaid-password-dialog.h"
#include "netaid-config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <sys/wait.h>

#include <simple-netaid/sbuf.h>
#include <simple-netaid/netproc.h>
#include <simple-netaid/iproute.h>

#include <libubox/blobmsg.h>
#include <libubox/blobmsg_json.h>

enum {
    COLUMN_ESSID = 0,
    COLUMN_ADDRESS,
    COLUMN_KEY,
    COLUMN_QUALITY,
    N_COLUMNS,
};

enum {
    PROP_0,
    PROP_INFO,
    N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES];

typedef struct {
    NetaidActiveWifis *aux;
    gchar *info;
} TaskData;

G_DEFINE_TYPE(NetaidActiveWifis, netaid_active_wifis, GTK_TYPE_TREE_VIEW)

static void netaid_active_wifis_finalize(GObject *object)
{
    G_OBJECT_CLASS(netaid_active_wifis_parent_class)->finalize(object);
}

static void netaid_active_wifis_constructed(GObject *object)
{
    NetaidActiveWifis *netaid_active_wifis;

    netaid_active_wifis = NETAID_ACTIVE_WIFIS(object);
    G_OBJECT_CLASS(netaid_active_wifis_parent_class)->constructed(object);
}

static void netaid_active_wifis_set_property(GObject *object,
        guint property_id, const GValue *value, GParamSpec *pspec)
{
    NetaidActiveWifis *self = NETAID_ACTIVE_WIFIS(object);

    switch (property_id) {
	case PROP_INFO:
		self->info = g_strdup(g_value_get_string(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    }
}

static void netaid_active_wifis_get_property(GObject *object, guint property_id,
        GValue *value, GParamSpec *pspec)
{
    NetaidActiveWifis *self = NETAID_ACTIVE_WIFIS(object);

    switch (property_id) {
    case PROP_INFO:
		g_free(self->info); 
		self->info = g_strdup(g_value_get_string(value));
		break;
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    }
}

static void netaid_active_wifis_class_init(NetaidActiveWifisClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);
    GtkContainerClass *container_class = GTK_CONTAINER_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    object_class->constructed = netaid_active_wifis_constructed;
    object_class->finalize = netaid_active_wifis_finalize;

    properties[PROP_INFO] = g_param_spec_string("info", NULL, NULL, NULL,
            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

    object_class->set_property = netaid_active_wifis_set_property;
    object_class->get_property = netaid_active_wifis_get_property;

    g_object_class_install_properties(object_class, N_PROPERTIES, properties);
}

static void on_password_cb(GtkDialog *unused, gpointer user_data)
{
    NetaidActiveWifis *self = user_data;

    GtkTreeModel *model;
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    
    const char *text = gtk_entry_get_text(GTK_ENTRY(self->dialog->entry));

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(self));
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(self));
    if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
        gchar *name;
        gtk_tree_model_get(model, &iter, COLUMN_ESSID, &name, -1);
        g_free(name);
    }
    gtk_widget_destroy(GTK_WIDGET(self->dialog));
}

static void on_destroy_cb(GtkWidget *widget, gpointer user_data)
{
    NetaidActiveWifis *self = NETAID_ACTIVE_WIFIS(user_data);
    gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(self), TRUE);
    self->dialog = NULL;
}

void on_press_event_cb(GtkTreeView *view, GtkTreePath *path,
		GtkTreeViewColumn *col, gpointer user_data)
{
    NetaidActiveWifis *self = NETAID_ACTIVE_WIFIS(view);
    GtkTreeModel *model;
    GtkTreeIter iter;

    model = gtk_tree_view_get_model(view);

    if (gtk_tree_model_get_iter(model, &iter, path)) {
        gchar *name = NULL;
        gboolean key;
        GtkWidget *window;

        gtk_tree_model_get(model, &iter, 
                           COLUMN_ESSID, &name, 
                           COLUMN_KEY, &key, -1);

        gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(self), FALSE);

        window = gtk_widget_get_ancestor(GTK_WIDGET(self), GTK_TYPE_WINDOW);
        
        self->dialog = netaid_password_dialog_new(GTK_WINDOW(window), name, key);

        g_signal_connect(self->dialog->connect_button, "clicked", G_CALLBACK(on_password_cb), self);
        g_signal_connect(self->dialog, "destroy", G_CALLBACK(on_destroy_cb), self);

        gtk_window_set_modal(GTK_WINDOW(self->dialog), TRUE);
        gtk_widget_show_all(GTK_WIDGET(self->dialog));

        g_free(name);
    }
}

static void netaid_active_wifis_init(NetaidActiveWifis *self)
{
    GtkTreeViewColumn *col;
    GtkCellRenderer *renderer;

    GtkListStore *store = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
			G_TYPE_BOOLEAN, G_TYPE_INT, -1);

    gtk_tree_view_set_model(GTK_TREE_VIEW(self), GTK_TREE_MODEL(store));
    self->store = store;  /* The ownership of the list moves to the window */

    /* destroy store automatically with view */
    g_object_unref(store);

    gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(self), TRUE);

    gtk_widget_set_hexpand(GTK_WIDGET(self), TRUE);

    renderer = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(col, renderer, TRUE);
    gtk_tree_view_column_add_attribute(col, renderer, "text", COLUMN_ESSID);
    gtk_tree_view_column_set_title(col, " ESSID");
    gtk_tree_view_append_column(GTK_TREE_VIEW(self),col);

    renderer = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(col, renderer, TRUE);
    gtk_tree_view_column_add_attribute(col, renderer, "text", COLUMN_ADDRESS);
    gtk_tree_view_column_set_title(col, " ADDRESS                   ");
    gtk_tree_view_append_column(GTK_TREE_VIEW(self),col);

    renderer = gtk_cell_renderer_toggle_new();
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(col, renderer, TRUE);
    gtk_tree_view_column_add_attribute(col, renderer, "active", COLUMN_KEY);
    gtk_tree_view_column_set_title(col, " KEY");
    gtk_tree_view_append_column(GTK_TREE_VIEW(self),col);
    
    renderer = gtk_cell_renderer_progress_new();
    g_object_set_data(G_OBJECT(renderer), "column", GINT_TO_POINTER(COLUMN_QUALITY));
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(self), -1,
                " QUALITY             ", renderer, "value", COLUMN_QUALITY, NULL);

    gtk_tree_view_set_activate_on_single_click(GTK_TREE_VIEW(self), TRUE);

    g_signal_connect(self, "row-activated", G_CALLBACK(on_press_event_cb), NULL);

    gtk_widget_set_size_request(GTK_WIDGET(self), 500, 300);

    gtk_widget_show_all(GTK_WIDGET(self));
}

static int parse_quality_percentage(const char *qual)
{
    if (!qual)
        return 0;

    int num, denom;

    if (sscanf(qual, "%d/%d", &num, &denom) == 2 && denom != 0)
        return (num * 100) / denom;

    return 0;
}

static gboolean netaid_active_wifis_scan_finish(NetaidActiveWifis *self,
        GAsyncResult *res, GError **error)
{
    g_return_val_if_fail(NETAID_IS_ACTIVE_WIFIS(self), FALSE);
    g_return_val_if_fail(g_task_is_valid(res, self), FALSE);

    return g_task_propagate_boolean(G_TASK(res), error);
}

static void task_ready_cb(GObject *source_object, GAsyncResult *res, gpointer user_data)
{  
    NetaidActiveWifis *self = NETAID_ACTIVE_WIFIS(source_object);
    GtkToolItem *refresh_item = NULL;
    GError *error = NULL;
    gboolean retval = FALSE;
    
    gboolean success = netaid_active_wifis_scan_finish(self, res, &error);

    GtkWidget *nb = gtk_widget_get_ancestor(GTK_WIDGET(self), NETAID_TYPE_NOTEBOOK);
    GtkWidget *win = gtk_widget_get_ancestor(GTK_WIDGET(self), GTK_TYPE_WINDOW);
    if (win) refresh_item = NETAID_WINDOW(win)->toolbar->refreshTb;

    if (nb) {
        gtk_spinner_stop(GTK_SPINNER(NETAID_NOTEBOOK(nb)->spinner));
        gtk_widget_hide(GTK_WIDGET(NETAID_NOTEBOOK(nb)->spinner));
    }
    
    if (refresh_item)
        gtk_widget_set_sensitive(GTK_WIDGET(refresh_item), TRUE);

    if (error) {
        g_warning("Scanning error: %s", error->message);
        g_error_free(error);
        return;
    }

    /* Parse json data */
    if (self->info && *self->info) {
        struct blob_buf b = {0};
        blob_buf_init(&b, 0);

        if (blobmsg_add_json_from_string(&b, self->info)) {
            if (self->store && GTK_IS_LIST_STORE(self->store)) {
                gtk_list_store_clear(self->store);
            } else {
                GtkListStore *new_store = gtk_list_store_new(N_COLUMNS,
                                               G_TYPE_STRING,  /* ESSID */
                                               G_TYPE_STRING,  /* ADDRESS */
                                               G_TYPE_BOOLEAN, /* KEY */
                                               G_TYPE_INT,     /* QUALITY */
                                               -1);
                g_set_object(&self->store, new_store);

                gtk_tree_view_set_model(GTK_TREE_VIEW(self), GTK_TREE_MODEL(new_store));
                g_object_unref(new_store);
            }

            enum { CELL_ADDR, CELL_QUAL, CELL_ENC, CELL_ESSID, __CELL_MAX };
            static const struct blobmsg_policy policy[__CELL_MAX] = {
                [CELL_ADDR]  = { .name = "Address", .type = BLOBMSG_TYPE_STRING },
                [CELL_QUAL]  = { .name = "Quality", .type = BLOBMSG_TYPE_STRING },
                [CELL_ENC]   = { .name = "key",     .type = BLOBMSG_TYPE_STRING },
                [CELL_ESSID] = { .name = "Essid",   .type = BLOBMSG_TYPE_STRING },
            };

            struct blob_attr *cell_attr;
            int rem;
            blobmsg_for_each_attr(cell_attr, b.head, rem) {
                struct blob_attr *tb[__CELL_MAX];
                blobmsg_parse(policy, __CELL_MAX, tb, blobmsg_data(cell_attr), blobmsg_len(cell_attr));

                if (tb[CELL_ADDR] && tb[CELL_ESSID]) {
                    GtkTreeIter iter;
                    const char *essid = blobmsg_get_string(tb[CELL_ESSID]);
                    const char *addr  = blobmsg_get_string(tb[CELL_ADDR]);
                    const char *qual  = blobmsg_get_string(tb[CELL_QUAL]);
                    const char *enc   = tb[CELL_ENC] ? blobmsg_get_string(tb[CELL_ENC]) : "off";

                    int quality_percent = 0;
                    int num, denom;
                    if (qual && sscanf(qual, "%d/%d", &num, &denom) == 2 && denom != 0)
                        quality_percent = (num * 100) / denom;

                    gtk_list_store_append(self->store, &iter);
                    gtk_list_store_set(self->store, &iter,
                        COLUMN_ESSID,   (essid && *essid) ? essid : "<hidden>",
                        COLUMN_ADDRESS, addr,
                        COLUMN_KEY,     g_ascii_strcasecmp(enc, "on"),
                        COLUMN_QUALITY, quality_percent,
                        -1);

                    retval = TRUE;
                }
            }
        }
        blob_buf_free(&b);
    }

    g_clear_pointer(&self->info, g_free);
    g_object_notify_by_pspec(G_OBJECT(self), properties[PROP_INFO]);
    
    if (retval)
        gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(self->store), 
                                             COLUMN_QUALITY, GTK_SORT_DESCENDING);

    gtk_widget_set_sensitive(GTK_WIDGET(refresh_item), TRUE);
}

static void task_thread_func(GTask *task, gpointer source_object,
        gpointer td, GCancellable *cancellable G_GNUC_UNUSED)
{
    NetaidActiveWifis *self = source_object;

    g_assert(NETAID_IS_ACTIVE_WIFIS(self));
    
    gchar *standard_output;
    gchar *standard_error;
    gint exit_status;
    gboolean ret __attribute__((unused));
    GError *error = NULL;
    GString *g_str = NULL;

    GtkWidget *notebook = gtk_widget_get_ancestor(GTK_WIDGET(self), GTK_TYPE_NOTEBOOK);
    GtkWidget *window = gtk_widget_get_ancestor(GTK_WIDGET(self), GTK_TYPE_WINDOW);
    GtkToolItem *refresh_tool_item = NETAID_WINDOW(window)->toolbar->refreshTb;

    gtk_spinner_start(GTK_SPINNER(NETAID_NOTEBOOK(notebook)->spinner));
    gtk_widget_show_all(GTK_WIDGET(NETAID_NOTEBOOK(notebook)->spinner));

    gtk_widget_set_sensitive(GTK_WIDGET(refresh_tool_item), FALSE);

    GValue info_gval = G_VALUE_INIT;
    g_value_init(&info_gval, G_TYPE_STRING);
    
    g_str = g_string_new(NETAID_WINDOW(window)->netaid_config->p.iwlist_scanning);
    netaid_config_parse_command(NETAID_WINDOW(window)->netaid_config, g_str, NULL, NULL, NULL);
    ret = g_spawn_command_line_sync(g_str->str, 
				&standard_output, &standard_error, &exit_status, &error);
    g_string_free(g_str, TRUE);

    g_value_set_string(&info_gval, standard_output);

    g_free(standard_output);
    g_free(standard_error);

    self->info = g_strdup(g_value_get_string(&info_gval));

    g_value_unset(&info_gval);

    g_task_return_boolean(task, TRUE);
}

void update_clicked_cb(GtkWidget *widget G_GNUC_UNUSED, gpointer user_data)
{
    GTask *task = NULL;
    NetaidActiveWifis *self = user_data;
    TaskData *task_data = NULL;

    g_assert(NETAID_IS_ACTIVE_WIFIS(self));

    g_object_notify_by_pspec(G_OBJECT(self), properties[PROP_INFO]);

    task = g_task_new(self, NULL, task_ready_cb, NULL);

    task_data = g_new0(TaskData, 1);
	task_data->aux = self;
    task_data->info = "";

    g_task_set_task_data(task, task_data, g_free);

    gtk_list_store_clear(GTK_LIST_STORE(self->store));

    g_task_run_in_thread(task, task_thread_func);

    /* g_task_run_in_thread takes a ref on the task until complete */
	g_clear_object(&task);
}

NetaidActiveWifis *netaid_active_wifis_new()
{
    NetaidActiveWifis *active_wifis = g_object_new(NETAID_TYPE_ACTIVE_WIFIS, NULL);
    return NETAID_ACTIVE_WIFIS(active_wifis);
}

