 /*
  * helpers.c
  * Copyright (C) Aitor Cuadrado Zubizarreta <aitor@genuen.org>
  * 
  * simple-netaid is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * simple-netaid is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */

#ifndef _GNU_SOURCE
   #define _GNU_SOURCE 1
#endif

// #include <sys/wait.h>
#include <sys/stat.h>
#include <assert.h>
#include <fcntl.h>
#include <ctype.h>

#include "def.h"
#include "helpers.h"
#include "sbuf.h"
#include "interfaces.h"
#include "ipaddr.h"

#define MAX_SIZE 128

short ifquery(const char *ifname)
{
    FILE *fp;
    short state;
    sbuf_t cmd;
      
    sbuf_init(&cmd);
    if (access("/sbin/ifquery", F_OK) == 0) {	
        sbuf_concat(&cmd, 2, "/sbin/ifquery --state ", ifname);
    } else if (access("/usr/sbin/ifquery", F_OK) == 0) {	
        sbuf_concat(&cmd, 2, "/usr/sbin/ifquery --state ", ifname);
    } else {
	    printf(_("Cannot find ifquery\n"));
	    exit( EXIT_FAILURE );
	}	
         
    fp = popen(cmd.buf, "r");
    if (!fp) {
        state=-1;
        goto Free;
    }
   
    state=0;
    if (fgetc(fp)!=EOF)
        state=1;
   
    pclose(fp);
   
Free:
    sbuf_free(&cmd);
   
    return state;
}

/**
 * Lee un PID desde un archivo (ej: /run/dhcpcd-eth0.pid)
 * Valida que el contenido sea numérico y el archivo no esté corrupto.
 */
static pid_t get_pid_from_file(const char *interface)
{
    char pid_path[PATH_MAX];
    char buffer[16];

    snprintf(pid_path, sizeof(pid_path), "/run/dhcpcd-%s.pid", interface);
    
    int fd = open(pid_path, O_RDONLY);
    if (fd < 0) return -1;

    ssize_t len = read(fd, buffer, sizeof(buffer) - 1);
    close(fd);

    if (len <= 0) return -1;
    buffer[len] = '\0';

    char *endptr;
    long pid = strtol(buffer, &endptr, 10);

    // Strtol error checking:
    // If buffer == endptr, no number was read.
    // If there is anything else besides spaces after the number, the format is questionable.
    if (buffer == endptr)
        return -1;

    // Optional: Validate PID range (0 to 32768 or 4M in modern systems)
    if (pid <= 0 || pid > 4194304)
        return -1;

    return (pid_t)pid;
}

void kill_all_processes(const char *ifname, const char *protected_ifname)
{
    DIR *dirp = NULL;
    struct dirent *dir = NULL;
    pid_t protected_pid = -1;
    const char *processes[] = { "dhclient", "dhcpcd", "udhcpc", "pump", 
                                "dhcp6c", "wpa_supplicant", NULL };
    
    if (protected_ifname)
        protected_pid = get_pid_from_file(protected_ifname);

    dirp = opendir("/proc");
    if (!dirp) {
        fprintf(stderr, "ERROR: Unable to open /proc: %m");
        return;
    }
    
    while ((dir = readdir(dirp)) != NULL) {
        char *endptr;
        // We try to convert the directory name directly to long
        long val = strtol(dir->d_name, &endptr, 10);

        // If endptr did not advance or did not reach the end of the string, it is not a valid PID
        if (dir->d_name == endptr || *endptr != '\0')
            continue;

        pid_t current_pid = (pid_t)val;

        if (current_pid == protected_pid || current_pid == 1 || current_pid == getpid())
            continue;

        char cmd_path[PATH_MAX];
        snprintf(cmd_path, sizeof(cmd_path), "/proc/%s/cmdline", dir->d_name);
        
        int fd = open(cmd_path, O_RDONLY);
        if (fd < 0) continue;

        char cmdline[1024];
        ssize_t len = read(fd, cmdline, sizeof(cmdline) - 1);
        close(fd);

        if (len <= 0) continue;
        cmdline[len] = '\0';

        char *binary_path = cmdline;
        char *binary_name = rindex(binary_path, '/');
        binary_name = binary_name ? binary_name + 1 : binary_path;

        bool match_proc = false;
        for (int i = 0; processes[i] != NULL; i++) {
            if (strcmp(binary_name, processes[i]) == 0) {
                match_proc = true;
                break;
            }
        }

        if (match_proc) {
            bool match_iface = false;
            char *arg = binary_path;
            
            while (arg < cmdline + len) {
                if (strcmp(arg, ifname) == 0) {
                    match_iface = true;
                    break;
                }
                arg += strlen(arg) + 1;
            }

            if (match_iface) {
                fprintf(stderr, "Killing %s (PID: %d) in interface %s",
                        binary_name, current_pid, ifname);
                if (kill(current_pid, SIGTERM) != 0)
                    kill(current_pid, SIGKILL);
            }
        }
    }
    closedir(dirp);
}
